/*
 * Cancan  --  mud client with telnet protocol
 *
 * Initially inspired to the Tintin client by Peter Unold, 
 * but Cancan contains no Tintin code.
 * The original Cancan was written by Mattias Engdeg}rd (Yorick) 
 * (f91-men@nada.kth.se) 1992-94,
 * and was greatly improved upon by Vivriel and Thuzzle and then
 * transformed from Cancan into Pow-wow by Cosmos who heavily worked
 * to make it yet more powerful.
 *
 * Then Bjorn Wesen (bjorn@sparta.lu.se) ported powwow to Win32,
 * and changed the name back to cancan since the name powwow clashed
 * with a semi-commercial but widespread internet phone product called powwow.
 *
 * As usual, all the developers are in debt to countless users 
 * for suggestions and debugging.
 *
 * This program is in the public domain.
 */

#ifndef RAND
#define random lrand48
#define srandom srand48
#endif

#define VERSION "1.97 (Yorick/Vivriel/Thuzzle/Ilie/Cosmos/Zarangal 1992-97)"
#define SAVEFILEVER 3
#define HELPNAME "cancan.hlp"

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <fcntl.h>
#include <signal.h>
#include <ctype.h>
#include <errno.h>
#include <time.h>
#include <sys/types.h>
#include <io.h>
#include <conio.h>
#include <process.h>
#include <memory.h>

#ifdef AIX
#include <sys/select.h>
#endif

#include "cancan.h"
#include "cmd.h"
#include "term.h"
#include "edit.h"
#include "xeval.h"

/*         function declarations                */
void mainlupe();
void exec_delays();
void printmotd();
void get_user_input();
void putbackcursor();
void put_word();
void set_param();
void restore_param();
int search_action();
int printstrlen();
void put_marks();
void wrap_print();
void smart_print();
void intrsig();
void termsig();
void pipesig();
void parse_commands();
void denewbiefy();
void unescape();
void showprompt();
void get_remote_output();
void writetofile();

char *strdup();

extern int errno, sys_nerr;
extern void bzero();
#define strncasecmp strncmp

/*        GLOBALS                */
char *helpname = HELPNAME;
long stat_var[NUMVAR+NUMPARAM];

struct {
  long num[NUMPARAM];
  char *str[NUMPARAM];
} def_param;

param_stack paramstk;
vars *var;                         /* pointer to all variables */

HANDLE outcons;  // global pointer to output handle
#define SCROLLBACKROWS 5000
PCHAR_INFO scrollback = 0;  // pointer to scrollback buffer
int scrollbackpos = 0; 
int scrollbackview = 0;

long received = 0;    /* amount of data received from remote host */
long sent = 0;        /* amount of data sent to remote host */

static char ready = 1;       /* 0 if cancan is parsing something */
static char confirm = 1;     /* 1 if just tried to quit */
char history_done = 0;       /* 1 if last command contained a #history */   
int internal;             /*  1 if internal echo, -1 if something sent to MUD,
			     -2 if #emulate used */
char echo_ext = 1;           /* 1 if text sent to MUD must be echoed */
char echo_int = 1;           /* 1 if internal messages are not suppressed */
char compactmode = 0;        /* 1 if to clear prompt between remote messages */
char debug = 0;              /* 1 if to echo every line before executing it */
char hostname[LINELEN], portnumber[LINELEN];
char deffile[LINELEN];             /* name and path of definition file */
char helpfile[LINELEN];            /* name and path of help file */
aliasnode *aliases;                /* head of alias list */
actionnode *actions;               /* head of action list */
marknode *markers;                 /* head of mark list */
int a_nice = 1;                    /* default priority of new actions/marks */
keynode *keydefs;                  /* head of key binding list */
delaynode *delays;                 /* head of delayed commands list */
delaynode *dead_delays;            /* head of dead-delayed commands list */
varnode *named_vars[2];            /* head of named variables list */
int num_named_vars[2];             /* number of named variables defined */
int max_named_vars = 100;          /* max number of named variables */
vtime now;                         /* current time */
vtime start_time;                  /* time of cancan timer startup */
vtime ref_time;                    /* time corresponding to #timer == 0 */
clock_t start_clock, cpu_clock;

consess con_sess[MAXFDSCAN];    /* connections indexed by socket descriptors */
volatile int sockfd = -1;                   /* current socket file descriptor;
				    * -1 means no socket */

static fd_set fdset;            /* set of descriptors to select() on */

char initstr[LINELEN];             /* initial string to send on connect */
char speedwalk = 0;                /* speedwalk on? */

int linemode;                      /* line mode flags (LM_* in cancan.h) */

/* for line editing */
int cols=80, lines=24;    /* screen size */
int col0;                 /* input line offset (= length of prompt) */
int line0;                /* screen line where the input line starts */
char edbuf[LINELEN];      /* line editing buffer */
int edlen;                /* length of current input line */
int pos = 0;              /* cursor position in line */
char promptstr[LINELEN];  /* entire prompt string (for redrawing) */
int promptlen = 0;        /* length of prompt string */
char promptintercept = 0; /* 1 if last prompt was intercepted by an #action */
char verbatim = 0;        /* 1 = don't expand aliases or process semicolons */
char prefixstr[LINELEN];  /* inserted in the editing buffer each time */
char inserted_next[LINELEN];/* inserted in buffer just once */
char flashback = 0;       /* cursor is on excursion and should be put back */
int excursion;            /* where the excursion is */
char wrap = 1;            /* 1 = word wrap active */
char edattrbeg[CAPLEN];   /* starting input line attributes */
char edattrend[CAPLEN];   /* ending input line attributes */

/* ugly debug macro hack, vt100-like only */
#define NOTE(param) (tty_printf("%c7%c[H%c[K",27,27,27),tty_printf param,tty_printf("%c8",27),tty_flush())

/* history buffer */
char *hist[MAXHIST];    /* saved history lines */
int curline = 0;        /* current history line */
int pickline = 0;        /* line to pick history from */

/* completion list */
char *words[WORDSKEPT];
int wordindex = 0;

FILE *capfile = (FILE *)NULL;        /* capture file or NULL */
FILE *recordfile = (FILE *)NULL;     /* record file or NULL */

FILE *logfile = (FILE *)NULL;        /* DEBUG log file or NULL */

#ifdef WIN32
int sockerr;
void InputThread(LPVOID vArg);
#endif

void main(argc, argv)
int argc; char *argv[];
{
    char *p;
    int i;

    /* initializations */
    initstr[0] = 0;
    memset(con_sess, 0, sizeof(con_sess));
    con_sess[1].active = -1;    /* no session opened */

    gettimeofday(&start_time, NULL);
    now.tv_sec  = ref_time.tv_sec  = start_time.tv_sec;
    now.tv_usec = ref_time.tv_usec = start_time.tv_usec;
    start_clock = cpu_clock = clock();
#ifndef NORANDOM
#ifndef WIN32
    srandom((int)now.tv_sec);
#endif
#endif

    paramstk.curr = -1;      /* stack is empty */

    if(p = getenv("CANCANHELP")) {
        strcpy(helpfile, p);
        if(helpfile[strlen(helpfile) - 1] != '/')
	    strcat(helpfile, "/");
        strcat(helpfile, helpname);
        if(_access(helpfile, 4) == -1 && !_access(helpname, 4))
		strcpy(helpfile, helpname);
    } else
        strcpy(helpfile, helpname);

    if (var = (vars *)malloc((NUMVAR+NUMPARAM+max_named_vars)*sizeof(vars))) {
	for (i=0; i<NUMVAR; i++) {
	    var[i].num = &stat_var[i];
	    var[i].str = NULL;
	}
	for (i=NUMVAR; i<NUMVAR+NUMPARAM; i++) {
	    var[i].num = &def_param.num[i];
	    var[i].str = def_param.str[i] = NULL;
	}
    }
    else
	syserr("malloc");
    
    if (argc == 1 || argc == 3) {
	default_keybindings();
	init_keybindings();
    } else if (argc == 2 || argc == 4) {
        /*
         * assuming first arg is definition file name
         * rules: if cancan dir is set it is searched first.
         *        if file doesn't exist, it is created there.
         * If a slash appears in the name, the cancan dir isn't used.
         * If three args, first is definition file name,
         * second and third are hostname and port number
         * (they overwrite the ones in definition file)
         */
        if(strchr(argv[1], '/') == NULL && (p = getenv("CANCANDIR"))) {
            strcpy(deffile, p);
            if(deffile[strlen(deffile) - 1] != '/') strcat(deffile, "/");
            strcat(deffile, argv[1]);
            if(_access(deffile, 6) == -1
               && !_access(argv[1], 6))
               strcpy(deffile, argv[1]);
        } else
            strcpy(deffile, argv[1]);
	
        if(_access(deffile, 6) == -1) {
            tty_printf("Creating %s\nHost name  :", deffile);
            fgets(hostname, LINELEN, stdin);
			if (hostname[0] == '\n')
				hostname[0] = '\0';
			else
				strtok(hostname, "\n");
            tty_printf("Port number:");
            fgets(portnumber, LINELEN, stdin);
			if (portnumber[0] == '\n')
				portnumber[0] = '\0';
			else
				strtok(portnumber, "\n");
            default_keybindings();
			init_keybindings();

            save_settings();
        } else {
            read_settings();
	}
    } 
    if (argc == 3 || argc == 4) {
        /* assume last two args are hostname and port number */
        strcpy(hostname, argv[argc - 2]);
        strcpy(portnumber, argv[argc - 1]);
    }

    settermcap();
    set_terminal();
    setsigs();

    tty_puts(clreoscr);
    tty_flush();

    {
	WORD wVersionRequested;
	WSADATA wsaData;
	
	wVersionRequested = MAKEWORD(2,1);
	if(WSAStartup(wVersionRequested,&wsaData))
	    syserr("Couldn't open WinSock!");
    }
    
    mainlupe();
}

/*
 * show current version
 */
void printver()
{
    tty_printf("Cancan version %s\nWin32 code/features by Bjorn Wesen (bjorn@sparta.lu.se)\n", VERSION);
}


/****************************************/
/* Separate thread to handle user input */
/****************************************/
void
InputThread(LPVOID vArg)
{
   int didget=0;
   HANDLE hConsoleInput;

   hConsoleInput=GetStdHandle(STD_INPUT_HANDLE);
   SetConsoleMode(hConsoleInput, ENABLE_PROCESSED_INPUT);

   while (TRUE) {
	  get_user_input();
   }
   _endthread();
}


/*
 * main loop.
 */
void mainlupe()
{
    fd_set readfds;
    int err;
    vtime timeoutbuf, *timeout;
    long sleeptime;
    COORD scsize = { 80, 512 };
    SMALL_RECT mysr = { 0, 0, 80, 25 };
    
    scrollback = malloc(sizeof(CHAR_INFO)*SCROLLBACKROWS*80);
    
    // get standard output handle for use by everything later
    outcons = GetStdHandle(STD_OUTPUT_HANDLE);
    //  SetConsoleScreenBufferSize(outcons, scsize);
    //  SetConsoleWindowInfo(outcons, TRUE, &mysr);
    
    FD_ZERO(&fdset);
    
    if (*hostname && *portnumber) {
	connect_sess("main", (*initstr ? initstr : NULL), hostname, 
		     atoi(portnumber));
	send_term_size();
    }
    
    tty_putc('\n');
    tty_gotoxy(col0 = 0, line0 = lines - 1);
    
    printver();
    
    tty_puts("Type #help for help.\n");
    confirm = 0;
    
    /*                main lupe        */
    
    /* establish a separate thread for user IO */
    SetThreadPriority((HANDLE)_beginthread(InputThread, 32768, NULL),
		      THREAD_PRIORITY_ABOVE_NORMAL);
    
    for(;;) 
    {
	readfds = fdset;
	gettimeofday(&now, NULL);

	sockfd = main_sess();
	exec_delays();
	flush_output_buffer();
	
	sleeptime = 0;
	if (delays) {
	    sleeptime = diff_vtime(&delays->when, &now);
	    if (!sleeptime)
		sleeptime = 1;    /* if sleeptime is less than 1 millisec,
				   * set to 1 millisec */
	}
	
	sleeptime = 1000;
	timeoutbuf.tv_sec = sleeptime / mSEC_PER_SEC;
	timeoutbuf.tv_usec = (sleeptime % mSEC_PER_SEC) * uSEC_PER_mSEC;
	timeout = &timeoutbuf;
	
	ready = 1;
	do {
	    /* if there are no fd's in readfds, select will return immediately without waiting
	     * hence we have to Sleep to avoid a busy loop in that case */
	    if(main_sess() < 0) {
		Sleep(500);
		err = 0;
	    } else {
		err = select(MAXFDSCAN, &readfds, NULL, NULL, timeout);
		sockerr = WSAGetLastError();
	    }
	} while(err == SOCKET_ERROR && sockerr == WSAEINTR);
	
	if (err == SOCKET_ERROR && sockerr != WSAEINTR)
	    syserr("select");
	ready = 0;

	if(flashback) putbackcursor();

	gettimeofday(&now, NULL);

	cpu_clock = clock();
	
	for (sockfd = 3; err && (sockfd < MAXFDSCAN); sockfd++)
	    if (FD_ISSET(sockfd, &readfds)) {
		err--;
		get_remote_output();
	    }
    }
}
/*
 * execute the delayed labels that have expired
 * and place them in the disabled delays list
 */
void exec_delays()
{
    delaynode *dying;
    char buf[LINELEN];
    
    while (delays && cmp_vtime(&delays->when, &now) <= 0) {
	dying = delays;           /* remove delayed command from active list */
	delays = dying->next;     /* and put it in the dead one  */
	
	add_node((defnode *)dying, (defnode **)&dead_delays, rev_time_sort);
	
	/* must be moved before executing delay->command
	 * and command must be copied in a buffer
	 * (can't you imagine why? The command may edit itself...)
	 */
	
	clearinpline(compactmode);
	if (!compactmode) {
	    tty_putc('\n');
	    col0 = 0;
	}
	
	
	if (echo_int)
	    tty_printf("#now [%s]\n", dying->command);
	
	error = 0; internal = 0;
	strcpy(buf, dying->command);
	parse_instruction(buf, 0, 0);
	history_done = 0;
	if (internal >= 0) {
	    showprompt();
	    col0 = printstrlen(promptstr);
	}
	else if (internal != -2)
	    promptlen = promptstr[0] = promptintercept = '\0'; 
	
	redraw_input();
    }
}

static char deleteprompt = 0;

/*
 * get output from the socket and process/display it.
 */
void get_remote_output()
{
    char buffer[BUFSIZE + 10];        /* allow for a terminating \0 later */
    char *buf;
    int got;
    got = read_socket(buffer, BUFSIZE - promptlen);
    if (!got) return;
    
    received += got;
    
#ifdef DEBUGCODE
    /* debug code to see in detail what strange codes come from the server */
    {
	int i; unsigned char c;
	tty_putc('{');
	for(i = 0; i < got; i++) 
	{
	    c = buffer[i];
	    if(c < ' ' || c > '~') tty_printf("[%d]", c); else tty_putc(c);
	}
	tty_printf("}\n");
    }
#endif
    
    if (con_sess[sockfd].active) {
	/* process only active sessions */
        /* line-at-a-time mode: process input in a number of ways */
	
        if(capfile)
	    writetofile(capfile, buffer, got);
	
	buf = buffer;
	
        /* 
	 * Code to merge lines from host that were splitted
	 * into different packets:
	 */
	
	if (!promptintercept && promptlen && *buf != '\n' && *buf != '\r') {
	    my_memmove(buf + promptlen, buf, got);
	    memcpy(buf, promptstr, promptlen);
	    got += promptlen;	  
	    tty_putc(' ');
	    tty_gotoxy(0, line0);
	    tty_printf("%s", clreoscr);
	    col0 = promptlen = promptstr[0] = promptintercept = '\0'; 
	}
	
        if (compactmode) {
	    if (*buf == '\n' || *buf == '\r') {     /* skip the first \n */
		buf++;
		got--;
		if (got && *buf != *buffer && (*buf == '\n' || *buf == '\r')) {
		    buf++;     /* some MUDs like to send "\n\r" or "\r\n" ... */
		    got--;
		}
		deleteprompt = 1;
	    }
        }
        else
	    deleteprompt = 0;
	
	if (got) {
	    clearinpline(deleteprompt);
	    deleteprompt = 0;
	    process_remote_input(buf, got);
	    redraw_input();
	}
    }
}

/*
 * process input from remote host:
 * detect special sequences, trigger actions, locate prompt, word-wrap,
 * print to stdout
 */
void process_remote_input(buf, size)
char *buf; int size;
{
    char onlyprompt = 1, *line;
    int i, mpilen = strlen(MPI);
    char buf2[LINELEN];
    
    if (buf[size - 1] == '\n' || buf[size - 1] == '\r')
	onlyprompt = 0;
    
    /* process block one line at a time: */
    i = 0;
    while(i < size) {
        line = buf + i;
        /*
         * some servers like to send NULs. Replace them with spaces, and hope
         * for the best. (I really should cut them out entirely)
         */
        for(; i < size && buf[i] != '\n'; i++)
            if (!buf[i])
	    buf[i] = ' ';
        buf[i] = '\0';        /* safe to do even if i == size, there is room */
	
        if(!strncmp(MPI, line, mpilen)) {
            i += process_message(line + mpilen, size - (line + mpilen - buf))
		+ mpilen;
        } else {
            if(i == size && printstrlen(line) < cols) {
                /* This is probably a prompt: strip any CR in the beginning. */
		
                while (*line == '\r') {
		    line++;
		    onlyprompt = 0;
		}
		
		if(onlyprompt && !strchr(line, '\b'))
		    strcat(promptstr, line);   /* MUME kludge */
		else
		    strcpy(promptstr, line);
		col0 = printstrlen(promptstr);
		promptlen = strlen(promptstr);
		
		promptintercept = search_action(promptstr); 
		/* Notice that prompts are still 
		 displayed even if intercepted... */
		
		put_marks(buf2, line);
		tty_printf("%s%s%s", edattrend, buf2, edattrbeg);
		tty_flush();
	    } else {
		onlyprompt = 0;
                if(!search_action(line)) {
                    if(line0 < lines - 1) line0++;
                    if (!is_main_sess())        /* sub connection */
                        tty_printf("##%s> ", con_sess[sockfd].id);
		    smart_print(line);
                }
                col0 = 0;
            }
        }
        i++;
    }
}

/*
 * Determine the printed length of a string. This is sometimes less than
 * the string length since it might contain escape sequences. Treated as such
 * are "<esc> [ <non-letters> <letter>", "<esc> <non-[>", "<control-char>".
 * This is not entirely universal but covers the most common cases (i.e. ansi)
 */
int printstrlen(s)
char *s;
{
    int l;
    enum { NORMAL, ESCAPE, BRACKET } state = NORMAL;
    for(l = 0; *s; s++) {
        switch(state) {
	  case NORMAL:
            if(*s == '\033') {
                state = ESCAPE;
            } else if(*s >= ' ') {
                l++;
            }
            break;
	    
	  case ESCAPE:
            state = (*s == '[') ? BRACKET : NORMAL;
            break;
	    
	  case BRACKET:
            if(isalpha(*s))
                state = NORMAL;
            break;
        }
    }
    return l;
}

/*
 * match action containing &1..&9 and $1..$9 and return actual params in
 * param array - return 1 if matched, 0 if not
 */
int match_action (pat, src)
char *pat, *src;
{
    char mpat[LINELEN], *npat=0, *npat2=0, *nsrc=0, *prm=0, c;
    char *tmp, *line=src;
    int mbeg = 0, mword = 0, p;
    
    for (p = 0; p < 10; p++) {
	*var[p+NUMVAR].str = '\0';
	*var[p+NUMVAR].num = 0;
    }
    
    if (*pat == '^') {
	pat++;
	mbeg = 1;  /* anchor match at line start */
    }
    if (*pat == '&' || *pat == '$')
	mbeg = - mbeg - 1;  /* pattern starts with '&' or '$' */
    
    while (pat && *pat) {
	if (((c=*pat) == '&' || c == '$')) {  /* &x matches a string */
	    /* $x matches a single word */
	    tmp = pat + 1;
	    if (isdigit(*tmp)) {
		p = 0;
		while (isdigit(*tmp) && p < NUMPARAM) {
		    p *= 10;
		    p += *tmp++ - '0';
		}
		if (p <= 0 || p >= NUMPARAM)
		    return 0;
		prm = var[p + NUMVAR].str;
		pat = tmp;
		if (c == '$')
		    mword = 1;
	    }
	}
	
	npat  = first_valid(pat, '&');
	npat2 = first_valid(pat, '$');
	if (npat2 < npat) npat = npat2;
	if (!*npat) npat = 0;
	
	if (npat) {
	    strncpy(mpat, pat, npat-pat);
	    mpat[npat - pat] = 0;
	}
	else
	    strcpy(mpat, pat);
	
	if (*mpat) {
	    nsrc = strstr(src, mpat);
	    if (!nsrc) return 0;
	    if (mbeg > 0) {
		if (nsrc != src) return 0;
		mbeg = 0;  /* reset mbeg to stop further start match */
	    } if (prm) {
		strncpy(prm, src, nsrc-src);
		prm[nsrc - src] = 0;
	    }
	} else if (prm) {
	    /* end of pattern space */
	    strcpy(prm, src);
	}
	
	/* post-processing of param */
	if (prm && mword) {
	    if (mbeg == -1) {
		/* unanchored '$' start, take last word */
		if ((p = my_strrstr(prm, DELIM, LINELEN)) != -1) {
		    tmp = prm + p;
		    do
			*prm++ = *++tmp;
		    while (*tmp);
		}
	    } else if (!pat[1]) {
		/* '$' at end of pattern, take first word */
		strtok(prm, DELIM);
	    } else {
		/* match only if param is single-worded */
		if (my_strstr(prm, DELIM, LINELEN) != -1) return 0;
	    }
	    mbeg = mword = 0;  /* reset match flags */
	}     
	src = nsrc + strlen(mpat);
	pat = npat;
    }
    strcpy(var[NUMVAR].str, line);
    return 1;
}

/*
 * Search for actions to trig on an input line. The line can't be trashed
 * since we want to print it on the screen later.
 * Return 1 if line matched to some action, 0 otherwise
 */
int search_action(line)
char *line;
{
    actionnode *p;
    params param;
    int ret = 0;
    
    set_param(&param);
    if (error) return 0;
    
    while(*line == '\r') line++;
    for(p = actions; p; p = p->next) {
        if(p->active && match_action(p->pattern, line)) {
            ret = 1;
	    error = 0;
	    internal = -1;
            parse_instruction(p->command, 0, 1);
	    history_done = 0;
            break;
	}
    }
    if (error!=DYN_STACK_UND_ERROR && error!=DYN_STACK_OV_ERROR)
	restore_param();
    return ret;
}

/*
 * match mark containing & and $ and return 1 if matched, 0 if not
 */

int match_mark(pat, src, start, end)
char *pat, *src, **start, **end;
{
    char mpat[LINELEN], *npat=0, *npat2=0, *nsrc=0, *prm=0, *endprm, *tmp, c;
    int mbeg = 0, mword = 0, p;
    
    *start = *end = 0;
    
    if (*pat == '&' || *pat == '$')
	mbeg = - 1;  /* pattern starts with '&' or '$' */
    
    while (pat && (c = *pat)) {
	if (c == '&' || c == '$') {  /* &x matches a string */
	    /* $x matches a single word */
	    pat++;
	    prm = src;
	    if (c == '$')
		mword = 1;
	}
	
	npat  = first_valid(pat, '&');
	npat2 = first_valid(pat, '$');
	if (npat2 < npat) npat = npat2;
	if (!*npat) npat = 0;
	
	if (npat) {
	    strncpy(mpat, pat, npat-pat);
	    mpat[npat - pat] = 0;
	} else
	    strcpy(mpat, pat);
	
	if (*mpat) {
	    nsrc = strstr(src, mpat);
	    if (!nsrc) return 0;
	    endprm = nsrc;
	    if (!*start)
		if (prm)
		*start = src;
	    else
		*start = nsrc;
	    *end = nsrc + strlen(mpat);
	} else if (prm)            /* end of pattern space */
	    *end = endprm = prm + strlen(prm);
	else
	    *end = src;
	
	
	/* post-processing of param */
	if (prm && mword) {
	    if (mbeg == -1 && (p = my_strrstr(prm, DELIM, endprm - prm - 1))
		!= -1)
		/* unanchored '$' start, take last word */
		*start = prm += p + 1;
	    else if (!*pat)
		/* '$' at end of pattern, take first word */
	        if ((p = my_strstr(prm, DELIM)) != -1)
		*end = endprm = prm + p;
	    else
		/* match only if param is single-worded */
		for (tmp = prm; tmp < endprm; tmp++)
		if (strchr(DELIM, *tmp))
		return 0;
	    mbeg = mword = 0;  /* reset match flags */
	}
	src = nsrc + strlen(mpat);
	pat = npat;
    }
    return 1;
}

/*
 * scan line for marked text and add correct marks
 */
void put_marks(dst, line)
char *dst, *line;
{
    marknode *np, *mfirst;
    char *p, *q, *first, *last;
    char begin[CAPLEN], end[CAPLEN];
    char *endline = line + strlen(line);
    int matchlen;
    
    if (*line) do {
	mfirst = 0;
	first = endline;
	for(np = markers; *line && np; np = np->next)
	    if (match_mark(np->pattern, line, &p, &q) && p<first) {
		
		/*          if((p = strstr(line, np->pattern)) && p<first) { */
		
		first = p;
		last = q;
		mfirst = np;
	    }
	
	if (mfirst) {
	    attr_string(mfirst->attrcode, begin, end);
	    
	    strncpy(dst, line, matchlen = first - line);
	    dst += matchlen;
	    line += matchlen;
	    
	    strcpy(dst, begin);
	    dst += strlen(begin);
	    
	    strncpy(dst, line, matchlen = last - first);
	    
	    dst += matchlen;
	    line += matchlen;
	    
	    strcpy(dst, end);
	    dst += strlen(end);
	}
    } while (mfirst);
    
    strcpy(dst, line);
}

void getsbline();
void scrollup();
void scrolldown();
void displaysb();

void getsbline()
{
    static COORD bufsz = { 80, SCROLLBACKROWS };
    CONSOLE_SCREEN_BUFFER_INFO cinf;
    static SMALL_RECT sr = { 0, 0, 79, 1 };
    COORD bufcord = { 0, scrollbackpos }; 
    GetConsoleScreenBufferInfo(outcons, &cinf);
    sr.Top = cinf.dwCursorPosition.Y - 1;
    sr.Bottom = sr.Top;
    ReadConsoleOutput(outcons, scrollback, bufsz, bufcord, &sr);
    scrollbackpos++;
    if(scrollbackpos >= SCROLLBACKROWS)
	scrollbackpos = 0;
}

void scrollup()
{
    scrollbackview-=25;
    displaysb();
}

void scrolldown()
{
    scrollbackview += 25;
    if(scrollbackview > 0)
	scrollbackview = 0;
    displaysb();
}

void displaysb()
{
    static COORD bufsz = { 80, SCROLLBACKROWS };
    COORD srccrd = { 0, 0 };
    SMALL_RECT dest = { 0, 0, 79, 24 };
    srccrd.Y = scrollbackpos + scrollbackview - 25;
    if(srccrd.Y < 0)
	srccrd.Y += SCROLLBACKROWS;
    if(srccrd.Y + 25 >= SCROLLBACKROWS) {
	dest.Bottom = SCROLLBACKROWS - srccrd.Y - 1;
	WriteConsoleOutput(outcons, scrollback, bufsz,
			   srccrd, &dest);
	dest.Top = dest.Bottom + 1;
	srccrd.Y = 0;
	dest.Bottom = 24;
    }
    WriteConsoleOutput(outcons, scrollback, bufsz, srccrd, 
		       &dest);
}


/*
 * write string to stdout, wrapping to next line if needed.
 */
void wrap_print(s)
char *s;
{
    char *p, c;
    char buf[LINELEN];
    int l, m;
    enum { NORMAL, ESCAPE, BRACKET } state;
    
    while(*s == '\r') {
	tty_putc(*s++);
    }
    l = printstrlen(s);
    if(!l) {
        /* empty line */
        tty_putc('\n'); tty_flush();
	getsbline();
        return;
    }
    
    while(l >= cols - col0 - 1) {
        p = buf; m = 0; state = NORMAL;
	while(m < cols - col0 - 1 && *s && *s != '\r') {
	    *p++ = c = *s++;
	    switch(state) {
	      case NORMAL:
		if(c == '\033')
		    state = ESCAPE;
		else if(c >= ' ')
		    m++, l--;
		break;
	      case ESCAPE:
		state = (c == '[') ? BRACKET : NORMAL;
		break;
	      case BRACKET:
		if(isalpha(c))
		    state = NORMAL;
		break;
	    }
	}
	
	*p = '\0';
	tty_printf("%s\n", buf);
	getsbline();
	col0 = 0;
    }
    
    if(l > 0) {
	tty_printf("%s\n",s);
	getsbline();
    }
}

void smart_print(line)
char *line;
{
    char buf[LINELEN];
    
    tty_printf("%s",edattrend);
    put_marks(buf, line);
    
    if (wrap)
	wrap_print(buf);
    else
	tty_printf("%s\n", buf);
    
    tty_puts(edattrbeg);
}

/*
 * write string to file, stripping \r
 */
void writetofile(f, str, len)
FILE *f; char *str; int len;
{
    for(; len--; str++) if(*str != '\r') putc(*str, f);
}

/*
 * read terminal input and send to parser.
 * decode keys that send escape sequences
 */
void get_user_input()
{
    unsigned char c;
    static char typed[CAPLEN];    /* chars typed so far (with partial match) */
    static unsigned int nchars = 0;                /* number of them */
    HANDLE inph = GetStdHandle(STD_INPUT_HANDLE);
    /* We have 4 possible line modes:
     * line mode, local echo: line editing functions in effect
     * line mode, no echo: sometimes used for passwords, no line editing
     * char mode, no echo: send a character directly, no local processing
     * char mode, local echo: extremely rare, do as above.
     */
    if(linemode & LM_CHAR) {
        int i;
        ReadConsole(inph, &c, 1, &i, 0);
	if(i != 1)
	    syserr("ReadConsole");
        if(write(sockfd, (char*)&c, 1) != 1)
            syserr("write single char to socket");
        if(!(linemode & LM_NOECHO)) {
	    tty_putc(c);
	    tty_flush();
        }
    }
#if 0
    else if(linemode & LM_NOECHO) {
	int i;
        /*
         * This is the no-echo, line-at-a-time mode usually used for
         * password entry. This is no good, since fgets will block until
         * the line is typed, but there is rarely anything to be read in
         * these cases anyway. (Wouldn't it be fun to display asterisks?)
         */
	tty_printf("Noecho linemode\n");
	SetConsoleMode(inph, ENABLE_LINE_INPUT | ENABLE_PROCESSED_INPUT);
	ReadConsole(inph, edbuf+strlen(edbuf), LINELEN-strlen(edbuf), &i, 0);
	SetConsoleMode(inph, ENABLE_PROCESSED_INPUT);
	//        fgets(edbuf + strlen(edbuf), LINELEN - strlen(edbuf), stdin);
	tty_printf("fgets returned\n");
        tty_putc('\n');
        edbuf[strlen(edbuf) - 1] = '\0';
        send_to_host(edbuf);
        typed[nchars = 0] = 0;
        edbuf[0] = 0; edlen = 0;
        col0 = promptlen = promptstr[0] = promptintercept = '\0';
    }
#endif
    else {
	/* normal mode (echo, line buffered) */
	keynode *p;
	int i;
	INPUT_RECORD my_ir;
	int my_key;
	
	/* use readconsoleinput to get all special keystrokes as well */
	ReadConsoleInput(inph, &my_ir, 1, &i);
	/* tty_printf("[%d]", c); */
	if(i != 1)
	    syserr("read from stdin");
	
	// ignore non-keypresses and ignore key-up events
	if(my_ir.EventType != KEY_EVENT)
	    return;
	if(!my_ir.Event.KeyEvent.bKeyDown)
	    return;
	
	/* extract key data */
	c = my_ir.Event.KeyEvent.uChar.AsciiChar;
	my_key = my_ir.Event.KeyEvent.wVirtualKeyCode;
	/* ignore shift, control and alt keys */
	if(my_key == VK_SHIFT || my_key == VK_MENU || my_key == VK_CONTROL)
	    return;
	/* tty_printf("[%d,%d]", my_key, c); */

#ifdef WIN32
	confirm = 0;
	if (flashback)
	  putbackcursor();
#endif
	
	if(c == 0 || (my_key >= VK_NUMPAD0 &&
		      my_key <= VK_DIVIDE)) { // special key?
		typed[nchars++] = '\033'; /* prepend esc */
	    c = my_key;
	    /* tty_printf("Spec key [%d]", c); */
	}
	/* search function key strings for match */
	typed[nchars++] = c; typed[nchars] = 0;
	for(p = keydefs; p && strncmp(typed, p->sequence, nchars); p=p->next);
	if(p) {
	    if(strlen(p->sequence) == nchars) {
#ifdef WIN32
		/* because we are on a separate thread */
		gettimeofday(&now);
#endif
		p->funct(p->call_data);
		typed[nchars = 0] = 0;
	    }
	} else {
	    insert_char((int)c);
	    typed[nchars = 0] = 0;
	}
    }
}
    
    
/*
 * Flash cursor at parentheses that matches c inserted before current pos
 */
void flashparen(c)
int c;
{
    int lev, i;
    for(i = pos - 1, lev = 0; i >= 0; i--)
        if(ISRPAREN(edbuf[i])) {
            lev++;
        } else if(ISLPAREN(edbuf[i])) {
            lev--;
            if(!lev)
                if(LPAREN(c) == edbuf[i])
		break;
	    else
		i = -1;
        }
    if(i >= 0) {
        optmove(CURLINE(pos), CURCOL(pos), CURLINE(i), CURCOL(i));
        tty_flush();
        flashback = 1;
        excursion = i;
    }
}

/*
 * put cursor back where it belongs
 */
void putbackcursor()
{
    optmove(CURLINE(excursion), CURCOL(excursion), CURLINE(pos), CURCOL(pos));
    tty_flush();
    flashback = 0;
}

/*
 * put string in history at current position
 * (string is assumed to be trashable)
 */
void put_in_history(str)
char *str;
{
    char *p;
    /* don't put passwords in history! */
    if(linemode & LM_NOECHO)
	return;
    if(hist[curline]) free(hist[curline]);
    hist[curline] = strdup(str);
    if(++curline == MAXHIST) curline = 0;
    /* split into words and put into completion list */
    for(p = strtok(str, DELIM); p; p = strtok((char*)NULL, DELIM)) {
	if(strlen(p) >= 3 && p[0] != '#') /* no commands/short words */
            put_word(p);
    }
}

/*
 * put word in word completion list
 */
void put_word(word)
char *word;
{
    if(words[wordindex]) free(words[wordindex]);
    words[wordindex] = strdup(word);
    if(++wordindex == WORDSKEPT) wordindex = 0;
}

/*
 * match and complete a word referring to the word list
 */
void complete_word()
{
    int i, j;
    
    if(pos) {
        /* find word start */
        for(j = pos; j > 0 && strchr(DELIM, edbuf[j - 1]) == NULL; j--);
        if(j == pos) return;
	
        if(edbuf[j] == '#' && pos - j > 1) {
            /* try to complete built-in command */
            char *p;
            for(i = 0; (p = commands[i].name); i++) {
                if(!strncmp(p, edbuf + j + 1, pos - j - 1))
                    break;
            }
            if(p)
                insertchars(p + pos - j - 1, strlen(p + pos - j) + 1);
        } else {
            /* scan word list for match */
            for(i = wordindex - 1; i != wordindex; i--) {
                if(i < 0) i += WORDSKEPT;
                if(words[i] == NULL) return;
                if(!strncasecmp(words[i], edbuf + j, pos - j)) break;
            }
            if(i != wordindex)
                insertchars(words[i] + pos - j, strlen(words[i] + pos - j));
        }
    }
}

/*
 * match and complete entire lines backwards in history
 */
void complete_line()
{
    int i;
    
    if(edlen) {
	for(i = curline - 1; i != curline; i--) 
	{
	    if(i < 0) i = MAXHIST - 1;
	    if(hist[i] && !strncmp(edbuf, hist[i], edlen)) break;
	}
	if(i != curline) 
	{
	    clearinpline(0);
	    strcpy(edbuf, hist[i]);
	    pos = edlen = strlen(edbuf);
	    redraw_input();
	}
    }
}

/*
 * put string into input line (will be redrawn next time)
 */
void to_input_line(str)
char *str;
{
    strcpy(inserted_next, str);
}

/* editor utility functions: */

/* insert one character first in a string */
void strins(s, c)
char *s; char c;
{
    char d;
    for(; *s; d = *s, *s++ = c, c = d);
    *s++ = c; *s = 0;
}

/* delete first character of a string */
void strdel(s)
char *s;
{
    while(*s) *s = s[1], s++;
}


/*
 * send a string to the main session on the remote host
 */
void send_to_main_host(str)
char *str;
{
    int osockfd = sockfd;  /* backup current socket fd */
    sockfd = main_sess();
    send_to_host(str);
    sockfd = osockfd;
}

/*
 * terminate cancan as cleanly as possible
 */
void exitcancan()
{
    if (capfile) fclose(capfile);
    if (recordfile) fclose(recordfile);
    tty_printf("#cancan stopped dancing.%s\n", edattrend);
    save_settings();
    show_stat();
    reset_terminal();
    if(scrollback)
	free(scrollback);
    WSACleanup();
    exit(0);
}

/*
 * set up our signal handlers
 */
void setsigs()
{
    signal(SIGINT, intrsig);
    signal(SIGTERM, termsig);
}

void pipesig()
{
    tty_printf("\n#no tobacco error.\n");
    exitcancan();
}

void intrsig()
{
    if (confirm) {
	tty_printf("\n#interrupted.\n");
	exitcancan();
    }
    
    tty_printf("\n#interrupted. Press again to quit\n");
    if (ready) {
	showprompt();
	redraw_input();
    }
    error = USER_BREAK;
    confirm = 1;
    signal(SIGINT, intrsig);
}

void termsig()
{
    tty_printf("\n#termination signal.\n");
    exitcancan();
}

/*
 * suspend ourselves
 */
void suspend_cancan()
{
    tty_printf("\n#I don't think your shell has job control.\n");
}

/*
 * print system call error message and terminate
 */
void syserr(msg)
char *msg;
{
    fprintf(stderr, "cancan system call error:\n %s (%d", msg, errno);
    if(errno > 0 && errno < sys_nerr)
	fprintf(stderr, ": %s", sys_errlist[errno]);
    fprintf(stderr, ")\n");
    reset_terminal();
    exit(1);
}

void set_param(altparam)
params *altparam;
{
    int i;
    
    if (paramstk.curr < MAX_STACK) {
	paramstk.p[++paramstk.curr] = (params *)var[NUMVAR].num;
	/* it works, because var[NUMVAR].num points to param.num[0],
	 * which is the first field of param.
	 */
	
	for(i=0; i<NUMPARAM; i++) {
	    *(var[i+NUMVAR].num = &altparam->num[i]) = 0;
	    *(var[i+NUMVAR].str = altparam->str[i]) = 0;
	}
    }
    else {
	print_error(error=DYN_STACK_OV_ERROR);
	paramstk.curr = -1;    /* reset stack to empty */
	for (i=0; i<NUMPARAM; i++) {
	    var[i+NUMVAR].num = &def_param.num[i];
	    var[i+NUMVAR].str = def_param.str[i];
	}
    }
}

void restore_param()
{
    int i;
    params *v;
    
    if (paramstk.curr >= 0) {
	v = paramstk.p[paramstk.curr--];
	
	for(i=0; i<NUMPARAM; i++) {
	    var[i+NUMVAR].num = &v->num[i];
	    var[i+NUMVAR].str = v->str[i];
	}
    }
    else {
	print_error(error=DYN_STACK_UND_ERROR);
	paramstk.curr = -1;    /* reset stack to empty */
	for (i=0; i<NUMPARAM; i++) {
	    var[i+NUMVAR].num = &def_param.num[i];
	    var[i+NUMVAR].str = def_param.str[i];
	}
    }
}

/*
 * parse input from user: calls parse_instruction for each instruction
 * in cmd_line.
 * silent = 1 if the line should not be echoed, 0 otherwise.
 */
void parse_user_input(line, silent)
char *line; char silent;
{
    do {
	line = parse_instruction(line, silent, 0);
    } while(!error && line && *line);
}

/*
 * execute walk if word is valid [speed]walk sequence -
 * return 1 if walked, 0 if not
 */
int parse_walk (word, silent, maponly)
char *word, silent, maponly;
{
    long strtol();
    char buf[16];
    unsigned int n = strlen(word);
    int is_main = is_main_sess();
    
    if (!is_main && !maponly && !speedwalk)
        return 0;
    if(!n || (n > 1 && !speedwalk && !maponly) ||
       !strchr("neswud", word[n - 1]) ||
       strspn(word, "neswud0123456789") != n)
        return 0;
    
    if (maponly) silent = 1;
    else if (speedwalk) internal = -1;
    buf[1] = '\0';
    while (*word) {
        if (!silent) tty_putc('[');
        if(isdigit(*word)) {
            n = strtol(word, &word, 10);
	    if (!silent)
	        tty_printf("%d", n);
	} else
            n = 1;
        if (!silent)
	    tty_putc(*word);
        while(n--) {
            *buf = *word;
            if (!maponly) {
	        send_to_host(buf);
	    }
            if(is_main || maponly)
                add_dir_to_map(*word);
        }
	if (!silent)
	    tty_puts("] ");
        word++;
    }
    if (!silent)
	tty_putc('\n');
    return !maponly;
}

/*
 * split str into words separated by DELIM, and place in
 * var[1+NUMVAR].str ... var[9+NUMVAR].str -
 * the whole str is put in var[0+NUMVAR].str
 */
char *split_words (str)
char *str;
{
    int i;
    char *prm;
    
    strcpy(var[NUMVAR].str, str);
    for (i=NUMVAR + 1; i < NUMPARAM+NUMVAR; i++) {
	*var[i].num = 0;
	prm = var[i].str;
	while (*str && strchr(DELIM, *str))
	    str++;
	while (*str && !strchr(DELIM, *str))
	    *prm++ = *str++;
	*prm = 0;
    }
    return str;
}

/*
 * copy first word of src into dst, and return pointer to second word of src
 */
char *split_first_word(dst, src)
char *dst, *src;
{
    char *tmp, *delim = DELIM;
    int what, got = 0;
    
    tmp = src = skipspace(src);
    
    if (!*src) {
	*dst='\0';
	return src;
    }
    
    do
	for  (what = 0; delim[what] && !got; what++)
	if (*tmp == delim[what])
	got = 1;
    while (!got && *++tmp);
    
    strncpy(dst, src, tmp-src);
    dst[tmp-src] = '\0';
    
    if (*tmp && *tmp != CMDSEP) tmp++;
    return tmp;
}

/*
 * Parse and exec the first instruction in 'line', and return pointer to the
 * second instruction in 'line' (if any).
 */
char *parse_instruction(line, silent, subs)
char *line; char silent, subs;
{
    aliasnode *np;
    char buf[LINELEN], substbuf[LINELEN];
    char *arg, *end, *ret;
    char last_is_sep = 1;
    
    if (error) return NULL;
    
    ret = get_next_instr(line);
    
    if (!ret || ret==line)  /* RETURN alone must be sent using a ';' */
	return ret;
    
    if (ret[-1] == CMDSEP)
	*--ret = '\0';
    else if (*ret) {
	tty_printf("#PANIC! Tried to execute: %s\n", line);
	error=SYNTAX_ERROR;
	return NULL;
    }
    else
	last_is_sep = 0;
    
    
    if (subs)
	subst_param(substbuf, line);
    else
	strcpy(substbuf, line);
    
    if (debug) tty_printf("#parsing: %s\n", substbuf); /* For run-time debugging */
    
    if (last_is_sep)
	*ret++ = CMDSEP;
    
    arg = skipspace(line = substbuf);
    
    if (arg[0] == '#' && arg[1] == '#') { /* send to other session */
	line = split_first_word(buf, arg + 2);
	
	if (!(sockfd = find_sess(buf))) {
	    tty_printf ("#no connection named '%s'", buf);
	    return NULL;
	}
	arg = skipspace(line);
    }
    
    if (*arg == '{') {    /* instruction contains a block */
	end = first_regular(line = arg + 1, '}');
	
	if (*end) {
	    *end = '\0';
	    parse_user_input(line, silent);
	    *end = '}';
	}
	else
	    print_error(error=MISSING_PAREN_ERROR);
    }
    else if (*arg == '#') {
	if (*(end = skipspace(arg + 1)) == '(') {    /* execute #() */
	    end++;
	    (void)xeval(NULL, &end, PRINT_NOTHING, 0);
	}
	else
	    parse_commands(buf, split_first_word(buf, arg + 1));
    } else {
	int oneword;
	/* initial spaces are NOT skipped this time */
	
	arg = split_first_word(buf, line); /* buf contains the first word,
					    arg points to arguments */
	if (!*arg) oneword = 1;
	else oneword = 0;
	
	if (np = *lookup_alias(buf)) {
	    params param;
	    set_param(&param);
	    if (error) return NULL;
	    
	    split_words(arg);  /* split argument into words
                                and place them in $0 ... $9 */
	    parse_instruction(np->subst, 0, 1);
	    
	    if (error!=DYN_STACK_UND_ERROR && error!=DYN_STACK_OV_ERROR)
		restore_param();
	    
	} else if (!oneword || !parse_walk(buf, silent, 0)) {
	    /* it is ok, parse_walk accepts only one word */
	    
	    unescape(line);
	    if(!silent && echo_ext) tty_printf("[%s]\n", line);
	    col0 = promptlen = promptstr[0] = promptintercept = '\0'; 
	    send_to_host(line);
	}
    }
    return ret;
}

/*
 * remove escapes (backslashes) from a string
 */
void unescape(s)
char *s;
{
    char *p;
    p = s;
    while(*s) {
	if(*s == '\\') {
	    s++;
	    while (*s == '\\')
		*p++ = *s++;
	    if (*s == NIL_CHAR) {
		*p++ = '\\';
		s++;
		continue;
	    }
	}
	if (*s != '\\')
	    *p++ = *s++;
    }
    *p = '\0';
}

/*
 * parse cancan's own commands
 */
void parse_commands(command, arg)
char *command, *arg;
{
    int i, j;
    
    /* disallow special commands on subsidiary connections */
    if (!is_main_sess()){
	tty_printf("#command not allowed on subsidiary connection\n");
	return;
    }
    
    /* assume output will be enough to make input line = last screen line */
    line0 = lines - 1;
    if(isdigit(*command) && (i = atoi(command)))
	if(i > 0)
	while(i--) (void)parse_instruction(arg, 1, 0);
    else
	tty_printf("#bogus repeat count\n");
    else {
	j = strlen(command);
	for(i = 0; commands[i].name; i++)
	    if(!strncmp(command, commands[i].name, j)) {
		if(commands[i].funct) {
		    tty_printf("%s", edattrend);
		    (*commands[i].funct)(arg);
		    return;
		}
	    }
	tty_printf("#unknown cancan command '%s'\n", command);
    }
}

/*
 * redisplay the prompt
 */
void showprompt()
{
    char buf[LINELEN];
    put_marks(buf, promptstr);
    tty_printf("%s%s%s", edattrend, buf, edattrbeg);
    tty_flush();
}

/*
 * cancel an editing session; does not free anything
 * (the child death signal handler will remove the session from the list)
 */
void cancel_edit(sp)
editsess *sp;
{
    char buf[LINELEN];
    char keystr[16];
    
#ifndef WIN32
    if(kill(sp->pid, SIGKILL) < 0)        /* Editicide */
	syserr("kill editor child");
#endif
    tty_printf("#killed %s (%d)\n", sp->descr, sp->key);
    sprintf(keystr, "C%d\n", sp->key);
    sprintf(buf, "%sE%d\n%s", MPI, (int) strlen(keystr), keystr);
    send_to_main_host(buf);
    sp->key = -1;        /* mark the child as cancelled */
}

/*
 * substitute $0..$9 and @0..@9 in a string 
 * (unless $ or @ is escaped with backslash)
 */
void subst_param(dst, src)
char *src, *dst;
{
    int i, subst;
    char *tmp, c;
    
    while(*src)  {
	subst = 0;
	if (*src == '$' || *src == '@') {
	    c = *src;
	    tmp = src + 1;
	    if (isdigit(*tmp)) {
		i = 0;
		while (isdigit(*tmp) && i < NUMPARAM) {
		    i *= 10;
		    i += *tmp++ - '0';
		}
		if (i < NUMPARAM) {
		    src = tmp;
		    subst = 1;
		    if (c == '$') {
			strcpy(dst, var[i+NUMVAR].str);
			if (*dst) while (*++dst) ;
		    } else {
			sprintf(dst, "%ld", *var[i+NUMVAR].num);
			while (*++dst) ;
		    }
		}
	    }
	} 
	if (!subst) {
	    if(*src == '\\') {
		src++;
		while (*src == '\\')
		    *dst++ = *src++;
		
		if (*src)
		    *dst++ = *src++;
	    }
	    else if (*src)
		*dst++ = *src++;
	}
    }
    *dst = '\0';
}

/*
 * return pointer to next non-blank char
 */
char *skipspace(p)
char *p;
{
    while(*p == ' ') p++;
    return p;
}

/*
 * read definitions from file
 */
void read_settings()
{
    char buf[LINELEN];
    FILE *f;
    char *p, *cmd;
    int version = 0;                /* save file version */
    
    if (!*deffile) {
	tty_printf("#warning: no savefile defined!\n");
	return;
    }
    
    f = fopen(deffile, "r");
    
    if(!f) syserr("fopen");
    
    while(aliases)
	delete_aliasnode(&aliases);
    while(actions)
	delete_actionnode(&actions);
    while (markers)
	delete_marknode(&markers);
    while (keydefs)
	delete_keynode(&keydefs);
    while (named_vars[0])
	delete_varnode(&named_vars[0], 0);
    while (named_vars[1])
	delete_varnode(&named_vars[1], 1);
    
    while(fgets(buf, LINELEN, f)) {
        cmd = skipspace(strtok(buf, "\n"));        /* strip lf */
        error = 0;
        if (!strncmp(cmd, "#(", 2)) {
            cmd += 2;
            (void)xeval(NULL, &cmd, PRINT_NOTHING, 1);
	}
        else if (*(p = first_regular(cmd, ' '))) {
            *p++ = '\0';
            if(!strcmp(cmd, "#savefile-version")) {
                version = atoi(p);
                if(version > SAVEFILEVER) {
                    tty_printf("#Warning: this cancan version is too old!\n");
                }
                else if(version < SAVEFILEVER) {
		    tty_printf("#Warning: config file is from an older version\n");
                }
            } else if(!strcmp(cmd, "#alias")) {
                parse_alias(p, 1);
            } else if(!strcmp(cmd, "#host")) {
                sscanf(p, "%s %s", hostname, portnumber);
            } else if(!strcmp(cmd, "#init")) {
                if (*(p=skipspace(p)) == '=')
		    strcpy(initstr, p + 1);
            } else if(!strcmp(cmd, "#action")) {
                parse_action(p, 1);
            } else if(!strcmp(cmd, "#mark")) {
                parse_marker(p, 1);
            } else if(!strcmp(cmd, "#bind")) {
                parse_bind_from_file(p);
            } else if(!strcmp(cmd, "#") && *p == '(') {
		++p;
		(void)xeval(NULL, &p, PRINT_NOTHING, 1);
            } else {
                tty_printf("#Invalid definition file command '%s'\n", buf);
            }
        } else if (*cmd)
            tty_printf("#bad definition file format\n");
    }
    fclose(f);
    if (version < 1) {
        tty_printf("#default keypad settings loaded\n");
        default_keybindings();
    }
    if (version < 2) {
        tty_printf("#default editing keys settings loaded\n");
        init_keybindings();
    }
}

void fail_msg()
{
    tty_printf("#error: can't write to definition file %s\n", deffile);
}

/*
 * save settings in definition file
 */
void save_settings()
{
    FILE *f;
    aliasnode *alp;
    actionnode *acp;
    marknode *mp;
    keynode *kp;
    varnode *vp;
    
    char buf[LINELEN], *buf2;
    int i, flag;
    
    if(!deffile || !*deffile) {
	tty_printf("#warning: no savefile defined!\n");
	return;
    }
    if((f = fopen(deffile, "w")) < 0) {
        fail_msg();
        return;
    }
    if (fprintf(f, "#savefile-version %d\n", SAVEFILEVER) < 0) {
        fail_msg();
        return;
    }
    if (fprintf(f, "#host %s %s\n", hostname, portnumber) < 0) {/* main host */
        fail_msg();
        return;
    }
    if(initstr && *initstr)
        if (fprintf(f, "#init =%s\n", initstr) < 0) {
	    fail_msg();
	    return;
	}
    for(alp = aliases; alp; alp = alp->next) {
        escape_specials(alp->name, buf2=buf);
        if (fprintf(f, "#alias %s=%s\n", buf, alp->subst) < 0) {
	    fail_msg();
	    return;
	}
    }
    for(acp = actions; acp; acp = acp->next) {
        escape_specials(acp->label, buf2=buf);
        while(*buf2) buf2++;
        *buf2++ = ' ';
        escape_specials(acp->pattern, buf2);
        while (*buf2) buf2++;
        if (fprintf(f, "#action >%c%s=%s\n", acp->active ? '+' : '-', buf, 
		    acp->command) < 0) {
	    fail_msg();
	    return;
	}
    }
    for(mp = markers; mp; mp = mp->next) {
        escape_specials(mp->pattern, buf);
        if (fprintf(f, "#mark %s=%s\n", buf, attr_name(mp->attrcode)) < 0) {
	    fail_msg();
	    return;
	}
    }
    for (flag = 0, i=0; i<NUMVAR; i++) {   /* save value of global variables */
        if(*var[i].num) {
            if (fprintf(f, "%s@%d = %ld", flag ? ", " : "#(", i-NUMVAR, 
			*var[i].num) < 0) {
	        fail_msg();
	        return;
	    }
            flag = 1;
        }
    }
    if (flag) fprintf(f, ")\n");
    for (i=0; i<NUMVAR; i++) {
        if(var[i].str && *var[i].str) {
	    escape_specials(var[i].str, buf);
	    if (fprintf(f, "#($%d = \"%s\")\n", i-NUMVAR, buf) < 0) {
	        fail_msg();
	        return;
	    }
	}
    }
    for (flag = 0, vp = named_vars[0]; vp; vp = vp->next) {
        if (vp->num) {
            if (fprintf(f, "%s@%s = %ld", flag ? ", " : "#(",
			vp->name, vp->num) < 0) {
	        fail_msg();
	        return;
	    }
            flag = 1;
	}
    }
    if (flag) fprintf(f, ")\n");
    for (vp = named_vars[1]; vp; vp = vp->next) {
        if (*vp->str) {
	    escape_specials(vp->str, buf);
            if (fprintf(f, "#($%s = \"%s\")\n", vp->name, buf) < 0) {
	        fail_msg();
	        return;
	    }
	}
    }
    for (i=0; i<NUMVAR; i++) {
        if(var[i].str && *var[i].str) {
	    escape_specials(var[i].str, buf);
	    if (fprintf(f, "#($%d = \"%s\")", i-NUMVAR, buf) < 0) {
	        fail_msg();
		return;
	    }
	}
    }
    
    for(kp = keydefs; kp; kp = kp->next) {
        escape_specials(kp->name, buf);
        if (fprintf(f, "#bind %s %s=%s\n", buf, seq_name(kp->sequence),
		    kp->funct==key_run_command ? kp->call_data :
		    internal_functions[lookup_edit_function(kp->funct)].name)
	    < 0 ) {
	    fail_msg();
	    return;
	}
    }
    fclose(f);
}

#ifdef NOSTRDUP
/*
 * some old libraries don't have strdup(), so...share and enjoy. 
 */
char *strdup(s)
char *s;
{
    char *p = (char*)malloc(strlen(s) + 1);
    if(p) strcpy(p, s);
    return p;
}
#endif


/*
 * Below are multiple-session support functions:
 *
 * note on con_sess[MAXFDSCAN]:
 *        con_sess[0].active holds total no. of connections
 *        con_sess[1].active holds main socket descriptor
 */

/*
 * return TRUE if current session being processed is main connection
 */
int is_main_sess ()
{
    return (sockfd == con_sess[1].active);
}



/*
 * return main session sockfd
 */
int main_sess ()
{
    return con_sess[1].active;
}

/*
 * return session's index or socket descriptor given id,
 * or 0 if null or invalid id is given
 */
int find_sess (id)
char *id;
{
    int i, n;
    
    for (i=3, n=con_sess[0].active; n; i++)
	if (con_sess[i].id) 
    {
        n--;
        if (!strcmp(con_sess[i].id, id))
	    return i;
    }
    return 0;
}



void show_host()
{
    char *h = NULL;
    
    if (!internal) internal = 1;
    if (*hostname)
	tty_printf("#default host: %s %s\n", hostname, portnumber);
    else
	tty_printf("#default host not defined.\n");
    
    if (sockfd)
	h = con_sess[sockfd].host;    
    if (h && *h)
	tty_printf("#current host: %s %d\n", h, con_sess[sockfd].port);
    else
	tty_printf("#current host not defined.\n");
}


/*
 * show list of open connections
 */
void show_connections()
{
    int i, n = con_sess[0].active;
    
    PRINT_INT ("#%s session%s opened%c\n", n ? "The following" : "No", 
	       n==1 ? " is" : "s are", n ? ':' : '.');
    
    
    for (i=3, n=con_sess[0].active; n; i++)
	if (con_sess[i].id) {
	    n--;
	    tty_printf("##%s\t:%sactive \t(%s %d)\n", con_sess[i].id, con_sess[i].active
		       ? "" : "non ", con_sess[i].host, con_sess[i].port);
	}
}

/*
 * connect another session
 */
void connect_sess (id, initstr, host, port)
char *id;        /* session id: "main" if main session */
char *initstr;        /* optional init string */
char *host; int port;  /* address of remote host */
{
    
    if (find_sess(id)) {
	PRINT_INT ("#session '%s' already open.\n", id);
	return;
    }
    
    /* dial the number by moving the right index in small circles */
    if ((sockfd = doconnect(host, port)) < 0) 
    {
	if (!strcmp(id, "main"))
	    syserr("#connect: host not responding");
	else
	    tty_printf("host not responding or too many open connections\n");
    }
    else 
    {
	FD_SET(sockfd, &fdset);         /* add socket to select() set */
	con_sess[0].active++;
	if (!strcmp(id, "main")) 
	{  /* main session */
	    con_sess[1].active = sockfd;  /* save main socket fd */
	    con_sess[sockfd].active = 1;
	}
	else
	{
	    if (!initstr || !*initstr)
		con_sess[sockfd].active = 1;  /* active only if no init */
	}
	
	con_sess[sockfd].id = strdup(id);
	con_sess[sockfd].host = strdup(host);
	con_sess[sockfd].port = port;

	gettimeofday(&now, NULL); /* an #at or #in may be inside initstr */

	if (initstr) 
	    parse_instruction(initstr, 0, 0);
    }
}

/*
 * disconnect session
 */
void disconnect_sess (id)
char *id;
{
    int sfd;
    
    if (id)
    {  /* #disconnect cmd */
	if (!(sfd = find_sess(id))) {
	    tty_printf("#no such connection\n");
	    return;
	}
	close(sfd);
    } 
    else
	sfd = sockfd;  /* connection closed by remote host */
    
    tty_printf("#Connection on %s closed.\n", con_sess[sfd].id);
    if (sfd == con_sess[1].active)  /* main connection closed */
	exitcancan();
    
    FD_CLR(sfd, &fdset);
    con_sess[0].active--;
    con_sess[sfd].active = 0;
    free(con_sess[sfd].id);
    con_sess[sfd].id = 0;
}

/*
 * toggle output display from another session
 */
void snoop_sess (id)
char *id;
{
    int sfd;
    
    sfd = find_sess(id);
    if(sfd)
        con_sess[sfd].active ^= 1;
}

