/*
 * Cmd.c
 * functions for cancan's built-in #commands
 *
 * (created: Finn Arne Gangstad (Ilie), Dec 25th, 1993)
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <ctype.h>
#include <time.h>
#ifndef WIN32
#include <sys/time.h>
#else
#include <io.h>
#define popen fopen
#define pclose fclose
#endif
#include <sys/types.h>
#include <errno.h>
#include <assert.h>
#ifdef AIX
#include <sys/select.h>
#endif
#include "cancan.h"
#include "cmd.h"
#include "term.h"
#include "edit.h"
#include "xeval.h"

void show_stat();
static void show_aliases();
static void show_actions();
static void show_marks();
static void show_keybinds();
void show_delaynode();
static void parse_bind();
static void define_new_key();
static void print_seq();
static char *read_seq();
static int get_one_char();
static void delete_action();
static void change_action();
static void add_new_action();
static void show_attr_syntax();
static void show_history();
static void exe_history();
static char *redirect();

static void cmd_help(), cmd_shell(),
 cmd_action(), cmd_add(), cmd_alias(), cmd_at(),
 cmd_beep(), cmd_bind(),
 cmd_cancel(), cmd_capture(), cmd_compact(), cmd_connect(), cmd_cpu(),
 cmd_do(), cmd_debug(),
 cmd_echo(), cmd_edit(), cmd_emulate(), cmd_exe(),
 cmd_file(), cmd_for(), cmd_hilite(), cmd_history(), cmd_host(),
 cmd_identify(), cmd_if(), cmd_in(), cmd_info(), cmd_init(),
 cmd_keyexe(), cmd_lines(), cmd_load(), cmd_map(), cmd_mark(), cmd_net(),
 cmd_nice(), cmd_prefix(), cmd_put(), cmd_xeval(), cmd_print(), cmd_quote(),
 cmd_resetkeys(), cmd_retrace(), cmd_record(),
 cmd_save(), cmd_send(), cmd_settimer(), cmd_snoop(), cmd_speedwalk(),
 cmd_stop(), cmd_time(), cmd_var(), cmd_ver(), cmd_while(), cmd_wrap(),
 cmd_zap();

#ifdef TELNETBUG
static void cmd_colour();
#endif

cmdstruct commands[] =
{
   {"help", "[keys|math|command]\tthis text, or help on specified topic",
       cmd_help},
   {"17", "command\t\t\trepeat 'command' 17 times", (void (*)())NULL},
#ifndef NOSHELL
   {"!", "shell-command\t\texecute a shell command using /bin/sh", cmd_shell},
#endif
   {"action", "[{>[+|-]|<|+|-|=}name] [{pattern|(expression)} [=[action]]]\n\t\t\t\tlist/delete/define actions", cmd_action},
   {"add", "{string|(expr)}\t\tadd the string to word completion list", 
       cmd_add},
   {"alias", "[name[=[text]]]\t\tlist/delete/define aliases", cmd_alias},
   {"at", "[name [(time-string) [command]]\tset time of delayed label",
      cmd_at},
   {"beep", "\t\t\t\tmake your terminal beep (like #print (*7))",
      cmd_beep},
   {"bind", "[edit|name[=[command]]]\tlist/delete/define key bindings",
      cmd_bind},
   {"cancel", "[number]\t\tcancel editing session", cmd_cancel},
   {"capture", "[filename]\t\tbegin/end of capture to file", cmd_capture},
#ifdef TELNETBUG
   {"color", "attr\t\t\tset default colours/attributes", cmd_colour},
#endif
   {"compact", "\t\t\ttoggles 'compact mode' (prompt deleted) on/off",
       cmd_compact},
   {"connect", "[session-id [initstr] [address port]\tconnect a new session",
      cmd_connect},
   {"cpu", "\t\t\t\tshow CPU time used by cancan", cmd_cpu},
   {"debug", "[on|off]\t\t\ttoggle debug mode (echo of every executed line)",
      cmd_debug},
   {"do", "(expr) command\t\trepeat 'command' (expr) times", cmd_do},
   {"echo", "[on|off]\t\t\ttoggle echo of text sent to mud on screen",
      cmd_echo},
   {"edit", "\t\t\t\tlist editing sessions", cmd_edit},
   {"emulate", "[<|!]{text|(string-expr)}\tprocess result as if received from\n\t\t\t\tremote host", cmd_emulate},
   {"exe", "[<|!]{text|(string-expr)}\texecute result as if typed from keyboard", cmd_exe},
   {"file", "[filename]\t\tset/show file name to save to/load from", cmd_file},
   {"for", "([init];check;[loop]) command\twhile 'check' is true exec 'command'", cmd_for},
   {"hilite", "[attr]\t\t\thighlight your input line", cmd_hilite},
   {"history", "[{number|(expr)}]\tlist/execute commands in history",
      cmd_history},
   {"host", "\t\t\t\tshow address of remote host", cmd_host},
   {"if", "(expr) instr1 [; #else instr2]\t if 'expr' is true execute 'instr1'\n\t\t\t\totherwise execute 'instr2'", cmd_if},
   {"identify", "[startact [endact]]\tsend MUME client identification",
       cmd_identify},
   {"in", "[label [(delay) [command]]]\tlist/delete/define delayed labels",
       cmd_in},
   {"init", "[=[command]]\t\tdefine command to execute on connect to host", 
       cmd_init},
   {"info", "[on|off]\t\t\ttoggle echo of cancan info messages on/off",
       cmd_info},
   {"key", "name\t\t\texecute the 'name' key binding", cmd_keyexe},
   {"lines", "[number]\t\t\tshow/set terminal lines", cmd_lines},
   {"load", "\t\t\t\tload cancan settings from file", cmd_load},
   {"map", "[-[number]|walksequence]\tshow/clear/edit (auto)map", cmd_map},
   {"mark", "[string[=[attr]]]\t\tlist/define markers", cmd_mark},
   {"net", "\t\t\t\tprint amount of data received from/sent to host", cmd_net},
   {"nice", "[{number|(expression)} [command]]\t\tset/show priority of new actions/marks",
       cmd_nice},
   {"prefix", "string\t\t\tprefix all lines with string", cmd_prefix},
   {"", "(expr)\t\t\tevaluate expression, trashing result", cmd_xeval},
   {"print", "[<|!][text|(expr)]\tprint string/result on screen, appending a \\n\n\t\t\t\tif no argument, prints value of variable $0", 
       cmd_print},
   {"put", "{text|(expression)}\tput text/result of expression in history",
       cmd_put},
   {"quote", "[on|off]\t\t\ttoggle verbatim-flag on/off", cmd_quote},
   {"record", "[filename]\t\tbegin/end of record to file", cmd_record},
   {"resetkeys", "\t\t\tclear all defined keys and reload default ones",
      cmd_resetkeys},
   {"retrace", "[number]\t\tretrace the last number steps", cmd_retrace},
   {"save", "\t\t\t\tsave cancan settings to file", cmd_save},
   {"send", "[<|!]{text|(string-expr)}\teval expression, sending result to the MUD", cmd_send},
   {"settimer", "(expr)\t\tsets internal timer to expr milliseconds",
      cmd_settimer},
   {"snoop", "session-id\t\ttoggle output display for session", cmd_snoop},
   {"speedwalk", "[on|off]\t\ttoggle speedwalk mode", cmd_speedwalk},
   {"stop", "\t\t\t\tremove all delayed commands from active list", cmd_stop},
   {"time", "\t\t\t\tprint current time and date", cmd_time},
   {"var", "variable [= [<|!]{string|(expression)} ]\twrite result into the variable", cmd_var},
   {"ver", "\t\t\t\tshow cancan version", cmd_ver},
   {"while", "(expr) instr\t\twhile 'expr' is true execute 'instr'",
       cmd_while},
   {"wrap", "\t\t\t\tturn word wrapping on/off (default is on)", cmd_wrap},
   {"zap", "session-id\t\t\tdisconnect a session", cmd_zap},
   {(char*)NULL, (char*)NULL, (void (*)())NULL}
};

int max_loop=10000;       /* maximum number of iterations in #for or #while */

/* anyone knows if ANSI 6429 talks about more than 8 colours? */
static char *colournames[] = {
    "black", "red", "green", "yellow", "blue", "magenta", "cyan", "white",
    "BLACK", "RED", "GREEN", "YELLOW", "BLUE", "MAGENTA", "CYAN", "WHITE"
};

static void cmd_nice(arg)
char *arg;
{
  int nnice = a_nice;
  arg = skipspace(arg);
  if (!*arg) {
    PRINT_INT ("#nice: %d\n", a_nice);
    return;
  }
  if (isdigit(*arg)) {
    a_nice = 0;
    while (isdigit(*arg)) {
      a_nice *= 10;
      a_nice += *arg++ - '0';
    }
  }
  else if (*arg++=='(') {
    char buf[LINELEN], *tmp = buf;
    int type;

    type = xeval(&tmp, &arg, PRINT_AS_NUM, 0);
    if (error)
      return;
    if (type!=TYPE_NUM) {
      print_error(error=NO_NUM_VALUE_ERROR);
      return;
    }    
    if (*arg++ != ')') {
        PRINT_INT ("#nice: ");
        print_error(error=MISSING_PAREN_ERROR);
        return;
    }
    a_nice = (int)*(long *)buf;
    if (a_nice<0)
      a_nice = 0;
  }
  arg = skipspace(arg);
  if (*arg) {
    parse_instruction(arg, 0, 0);
    a_nice = nnice;
  }
}

static void cmd_init(arg)
char *arg;
{
  arg = skipspace(arg);

  if (*arg == '=')
     if (*++arg) {
       strcpy(initstr, arg);
       if (echo_int) {
	 PRINT_INT ("#init: %s\n", initstr);
       }
     }
     else {
       *initstr = '\0';
       if (echo_int) {
         PRINT_INT ("#init cleared\n");
       }
     }
  else {
     char buf[LINELEN];
     strcpy(buf, "#init =");
     strcat(buf, initstr);
     to_input_line(buf);
  }
}

/*ARGSUSED*/
static void cmd_host(arg)
char *arg;
{
  show_host();
}

/*ARGSUSED*/
static void cmd_beep(arg)
char *arg;
{
  tty_puts ("\007"); tty_flush();
}

static void cmd_hilite(arg)
char *arg;
{
    int attr;

    arg = skipspace(arg);
    attr = parse_attributes(arg);
    if(attr < 0) {
       PRINT_INT ("#attribute syntax error\n");
       if (echo_int) show_attr_syntax();
    } else {
       attr_string(attr, edattrbeg, edattrend);
       if (echo_int) {
          PRINT_INT ("#input highlighting is now %so%s%s.\n",
              edattrbeg, (attr == NOATTRCODE) ? "ff" : "n", edattrend);
       }
    }
}

/*
 * return token number for attribute word:
 * this is colour number for colours, TOK_xxx for effects, TOK_ON for "on",
 * or -1 for unrecognized word.
 */
int attr_token(word)
char *word;
{
#ifdef WIN32
#define strcasecmp strcmp
#endif
    int i;
    for(i = 0; i < COLOURS && strcmp(word, colournames[i]); i++);
    if(i < COLOURS) return i;
    if(strcasecmp(word, "bold") == 0)
       return BOLD;
    else if(strcasecmp(word, "underline") == 0)
       return UNDERLINE;
    else if(strcasecmp(word, "inverse") == 0
            || strcasecmp(word, "reverse") == 0)
        return INVERSE;
    else if(strcasecmp(word, "on") == 0)
        return TOK_ON;
    else if(strcasecmp(word, "none") == 0)
        return NOATTRIB;
    else
        return TOK_INVALID;
}

/*
 * parse attribute description in line.
 * Return attribute if successful, -1 otherwise.
 */
int parse_attributes(line)
char *line;
{
    char *p;
    int t, fg;
    p = strtok(line, " ");
    
    if(p == NULL) {
        return NOATTRCODE;
    }
    if((t = attr_token(p))== TOK_INVALID) return -1;
    if(t < COLOURS) {
        fg = t;
        if((p = strtok(NULL, " ")) == NULL) return ATTRCODE(fg, NOATTRIB);
        if((t = attr_token(p)) != TOK_ON) return -1;
    } else if(t != TOK_ON) {
        return ATTRCODE(t, NOATTRIB);
    } else
        fg = NOATTRIB;
    if((p = strtok(NULL, " ")) == NULL) return -1;
    t = attr_token(p);
    if(t < 0 || t >= COLOURS) return -1;
    return ATTRCODE(fg, t);
}

/*
 * display attribute syntax
 */
static void show_attr_syntax()
{
    int i;
    PRINT_INT ("#attribute syntax:\n\tOne of:\tbold, underline, inverse, none\n");
    tty_printf("\tor:\t[foreground] [ON background]\n");
    tty_printf("\tColours: ");
    for(i = 0; i < COLOURS; i++) {
        tty_printf("%s%s", colournames[i], (i == LOWCOLOURS - 1) ? "\n\t\t" : 
	       (i == COLOURS - 1) ? "\n" : ",");
    }
}

/*
 * return a static pointer to name of given attribute code
 */
char *attr_name(attrcode)
int attrcode;
{
    static char name[80];        /* should be enough */
    int fg = FOREGROUND(attrcode), bg = BACKGROUND(attrcode);

    name[0] = 0;
    if(fg < COLOURS)
        strcpy(name, colournames[fg]);
    else if(fg == BOLD)
        strcpy(name, "bold");
    else if(fg == UNDERLINE)
        strcpy(name, "underline");
    else if(fg == INVERSE)
        strcpy(name, "inverse");
    else if (fg == NOATTRIB)
        strcpy(name, "none");
    if(bg < NOATTRIB) {
        if(name[0]) strcat(name, " ");
        strcat(name, "on ");
        strcat(name, colournames[bg]);
    }
    else if(bg > NOATTRIB)
        name[0] = 0;   /* error! */
    return name;
}

/*
 * put escape sequences to turn on/off an attribute in given buffers
 */
void attr_string(attrcode, begin, end)
int attrcode; char *begin, *end;
{
    int fg = FOREGROUND(attrcode), bg = BACKGROUND(attrcode);
    *begin = *end = '\0';
    if(fg < COLOURS) {
        if(bg < COLOURS)
	    sprintf(begin, "\033[;%c%d;%s%dm",
		    fg<LOWCOLOURS ? '3' : '9', fg % LOWCOLOURS,
		    bg<LOWCOLOURS ? "4" : "10", bg % LOWCOLOURS);
        else
            sprintf(begin, "\033[%c%dm",
		    fg<LOWCOLOURS ? '3' : '9', fg % LOWCOLOURS);
    } else if(bg < COLOURS)
        sprintf(begin, "\033[%s%dm", 
		bg<LOWCOLOURS ? "4" : "10", bg % LOWCOLOURS);
    else if(fg == BOLD)
        if(modebold[0])
            strcpy(begin, modebold);
        else {
            strcpy(begin, modestandon);
            strcpy(end, modestandoff);
        }
    else if(fg == UNDERLINE)
        if(modeuline[0])
            strcpy(begin, modeuline);
        else {
            strcpy(begin, modestandon);
            strcpy(end, modestandoff);
        }
    else if(fg == INVERSE)
        if(modeinv[0])
            strcpy(begin, modeinv);
        else {
            strcpy(begin, modestandon);
            strcpy(end, modestandoff);
        }
           /* 'none' attribute is automatic... */
    if(attrcode != NOATTRCODE && !*end)
        strcpy(end, modenorm);
}

static void cmd_mark(arg)
char *arg;
{
    if(!*arg)
        show_marks();
    else
        parse_marker(arg, 0);
}

/*ARGSUSED*/
static void cmd_resetkeys(arg)
char *arg;
{
    while (keydefs)
        delete_keynode(&keydefs);
    init_keybindings();
    default_keybindings();
    settermcap();
}


static void cmd_bind(arg)
char *arg;
{
    arg = skipspace(arg);
    if(!*arg)
        show_keybinds(0);
    else if (!strcmp(arg, "edit"))
        show_keybinds(1);
    else
        parse_bind(arg);
}

/*
 * list keyboard bindings
 */
static void show_keybinds(edit)
char edit;
{
    keynode *p;
    int count = 0;
    for(p = keydefs; p; p = p->next) {
	if (edit != (p->funct == key_run_command)) {
	  if(!count)
	    if (edit) {
	      PRINT_INT ("#line-editing keys:\n");
	    } else {
	      PRINT_INT ("#user-defined keys:\n");
	    }
	  tty_printf("#bind %s %s=%s\n", p->name, seq_name(p->sequence),
	       p->funct == key_run_command ? p->call_data :
	       internal_functions[lookup_edit_function(p->funct)].name);
	  ++count;
	}
    }
    if(!count) {
        PRINT_INT ("#No key bindings defined right now.\n");
    }
}

static void cmd_keyexe(arg)
char *arg;
{
    keynode *p, *q=NULL;

    if (!*arg) return;

    for(p = keydefs; p && !q; p = p->next) {
        if (p->funct == key_run_command && !strcmp(p->name, arg))
            q = p;
    }
    if (q)
        parse_instruction(q->call_data, 0, 0);
    else {
        PRINT_INT("#no such key: %s\n", arg);
    }
}

/*
 * parse the #bind command from a definition file
 */
void parse_bind_from_file(arg)
char *arg;
{
    char keyseq[CAPLEN], *q = keyseq, *p, *name = arg;
    int escape = 0, caret = 0, function;
    keynode **kp;

    p = strchr(arg, ' ');
    if(!p) {
        PRINT_INT ("#bad definition file syntax: #bind %s\n", arg);
        return;
    }
    *(p++) = '\0';
    for(; *p; p++) {
        if(escape) {
            *(q++) = *p;
            escape = 0;
        } else if(caret) {
            *(q++) = CONTROL(*p);
            caret = 0;
        } else if(*p == '\\')
            escape = 1;
        else if(*p == '^')
            caret = 1;
        else if(*p == '=')
            break;
        else
            *(q++) = *p;
    }
    if(!*p) {
        PRINT_INT ("#bad definition file syntax\n");
        return;
    }
    *q = '\0';
    p++;
    kp = lookup_key(name);
    if (!kp || !*kp)
      if (function = lookup_edit_name(p))
	add_keynode(name, keyseq, internal_functions[function].funct, NULL);
      else
	add_keynode(name, keyseq, key_run_command, p);
}

/*
 * parse the argument of the #bind command (interactive)
 */
static void parse_bind(arg)
char *arg;
{
    char *p, *command;
    char *name = arg;
    keynode **np;
    int function;

    p = strchr(arg, '=');
    if(p) {
        *(p++) = '\0';
        np = lookup_key(name);
        if(*p) {
            command = p;
            if(*np) {
                if((*np)->funct == key_run_command)
                    free((*np)->call_data);
		if (function = lookup_edit_name(command)) {
		    (*np)->call_data = NULL;
		    (*np)->funct = internal_functions[function].funct;
		} else {
                    (*np)->call_data = strdup(command);
		    (*np)->funct = key_run_command;
		}
                if (echo_int) {
                    PRINT_INT ("#redefined key: %s=%s\n", name, command);
                }
            } else
                define_new_key(name, command);
        } else {
            if(*np) {
                if (echo_int) {
                    PRINT_INT ("#deleting key binding: %s=%s\n", name,
		       (*np)->call_data);
                 }
                 delete_keynode(np);
            } else {
                PRINT_INT ("#no such key: %s\n", name);
            }
        }
    } else {
        np = lookup_key(name);
        if(*np) {
	    char buf[LINELEN];
            sprintf(buf, "#bind %s=%s", name,
		 (*np)->funct == key_run_command ? (*np)->call_data : 
		 internal_functions[lookup_edit_function((*np)->funct)].name);
	    to_input_line(buf);
        } else {
            PRINT_INT ("#no such key: %s\n", name);
        }
    }
}

/*
 * define a new key, prompting user to press it
 */
static void define_new_key(name, command)
char *name, *command;
{
    char c, keyseq[] = "\033x";
    keynode *p;
    int seqlen = 2, function;
    INPUT_RECORD my_ir;
    int my_key, i;

    tty_printf("#please press the key you want to redefine: ");
    tty_flush();

	// use readconsoleinput to get all special keystrokes as well
	do {
	    ReadConsoleInput(GetStdHandle(STD_INPUT_HANDLE),
			     &my_ir, 1, &i);
	    if(i != 1)
	      syserr("read from stdin");
	    
	    // ignore non-keypresses and ignore key-up events
	    if(my_ir.EventType != KEY_EVENT)
	      continue;
	    if(!my_ir.Event.KeyEvent.bKeyDown)
	      continue;
	    my_key = my_ir.Event.KeyEvent.wVirtualKeyCode;
	    if(my_key == VK_SHIFT || 
	       my_key == VK_MENU ||
	       my_key == VK_CONTROL)
	      continue;
	    break;
	} while(1);
    
    c = my_ir.Event.KeyEvent.uChar.AsciiChar;
    
    tty_printf("\nSelected virtual code %x (normal %x).\n", my_key, c);
    
    if(c && !(my_key >= VK_NUMPAD0 && my_key <= VK_DIVIDE) ) {
        PRINT_INT ("#That is not a redefinable key.\n");
    } else {
	keyseq[1] = my_key;
        function = lookup_edit_name(command);

        for(p = keydefs; p; p = p->next) {
	    if(!strncmp(p->sequence, keyseq, seqlen)) {
		PRINT_INT ("Key already bound as: %s=%s\n", p->name, 
			   p->funct == key_run_command ? p->call_data :
			   internal_functions[lookup_edit_function(p->funct)].name);
		return;
	    }
	}
        if (function)
	  add_keynode(name, keyseq, internal_functions[function].funct, NULL);
        else
	  add_keynode(name, keyseq, key_run_command, command);

        if (echo_int) {
	    PRINT_INT ("#new key binding: %s %s=%s\n", name, seq_name(keyseq),
		       command);
        }
    }
}

/*
 * read a single escape sequence from the keyboard; return static pointer
 */
static char *read_seq()
{
    static char seq[CAPLEN];
    int i = 0, tmp;

    seq[i++] = get_one_char(0);
    while(i < CAPLEN - 1 && (tmp = get_one_char(KBD_TIMEOUT)) >= 0)
        seq[i++] = tmp;
    seq[i] = '\0';
    return seq;
}

/*
 * read a single character from stdin, with timeout in microseconds.
 * timeout == 0 means wait indefinitely (no timeout).
 * return char or -1 if timeout was reached.
 */
static int get_one_char(timeout)
int timeout;
{
    return getchar();
}

/*
 * print an escape sequence human-readably
 */
static void print_seq(seq)
char *seq;
{
    char ch;
    
    while(ch = *(seq++)) {
        if(ch == '\033') {
            tty_printf("esc ");
            continue;
        }
        if(ch < ' ') {
            tty_printf("^");
            ch |= '@';
        }
        if(ch == ' ')
            tty_printf("space ");
        else
            tty_printf("%c ", ch);
    }
}

/*
 * return a static pointer to escape sequence made printable (for use in
 * definition-files
 */
char *seq_name(seq)
char *seq;
{
    static char buf[LINELEN];
    char *p = buf;
    /*
     * rules: escape =\^ with \. control chars are written as ^X.
     */
    for(;*seq; seq++) {
        if(*seq == '=' || *seq == '\\' || *seq == '^')
            *(p++) = '\\';
        if(*seq < ' ') {
            *(p++) = '^';
            *(p++) = *seq | '@';
        } else
            *(p++) = *seq;
    }
    *p = '\0';
    return buf;
}

static void cmd_map(arg)
char *arg;
{
    arg = skipspace(arg);
    if (!*arg)  /* show map */
        show_automap();
    else if (*arg == '-')  /* retrace steps without walking */
        retrace_steps(atoi(arg + 1), 0);
    else
        parse_walk(arg, 1, 1);
}

static void cmd_retrace(arg)
char *arg;
{
    retrace_steps(-atoi(arg), 1);
}

static void cmd_alias(arg)
char *arg;
{
    if(!*arg)
        show_aliases();
    else
        parse_alias(arg, 0);
}

static void cmd_add(arg)
char *arg;
{
  char buf[LINELEN];
  char *tmp = skipspace(arg);

  if (*tmp=='(') {
    arg = tmp + 1;
    tmp = buf;
    (void)xeval(&tmp, &arg, PRINT_AS_TEXT, 0);
    if (error) return;
    if (*arg != ')') {
      PRINT_INT ("#add: ");
      print_error(error=MISSING_PAREN_ERROR);
      return;
    }
    arg = buf;
  }
  while (*arg) {
    arg = split_first_word(buf, arg);
    put_word(buf);
  }
}
static void cmd_put(arg)
char *arg;
{
  char buf[LINELEN];
  char *tmp = skipspace(arg);

  if (*tmp=='(') {
    arg = tmp + 1;
    tmp = buf;
    (void)xeval(&tmp, &arg, PRINT_AS_TEXT, 0);
    if (error) return;
    if (*arg != ')') {
      PRINT_INT ("#put: ");
      print_error(error=MISSING_PAREN_ERROR);
      return;
    }
    arg = buf;
  }
  if (*arg)
    put_in_history(arg);
}

static char *redirect(arg, buf, kind, name, also_num, start, end)
char *arg, *buf, *kind, *name, also_num;
long *start, *end;
{
  char *tmp = skipspace(arg), k;
  char buf2[LINELEN];
  int type, i;

  k = *tmp;
  if (k == '!' || k == '<')
    arg = ++tmp;
  else
    k = 0;

  *start = *end = 0;

  if (*tmp=='(') {
    arg = tmp + 1;
    tmp = buf;
    type = xeval(&tmp, &arg, PRINT_AS_TEXT, 0);
    if (error)
        return NULL;
    if (type!=TYPE_TXT && !also_num) {
      PRINT_INT ("#%s: ", name);
      print_error(error=NO_STRING_ERROR);
      return NULL;
    }
    for (i=0; i<2; i++) if (*arg == CMDSEP) {
        arg++;
	if (!i && *arg == CMDSEP) {
	  *start = 1;
	  continue;
	}
	else if (i && *arg == ')') {
	  *end = LONG_MAX;
	  continue;
	}

	tmp = buf2;
	type = xeval(&tmp, &arg, PRINT_AS_NUM, 0);
	if (error)
	    return NULL;
	if (type != TYPE_NUM) {
	  PRINT_INT ("#%s: ", name);
	  print_error(error=NO_NUM_VALUE_ERROR);
	  return NULL;
	}
	if (i)
	  *end = *(long*)buf2;
	else
	  *start = *(long*)buf2;
    }
    if (*arg != ')') {
        PRINT_INT ("#%s: ", name);
        print_error(error=MISSING_PAREN_ERROR);
        return NULL;
    }
    arg = buf;
    if (!*start && *end)
	*start = 1;
  }
  else
    unescape(arg);

  *kind = k;
  return arg;
}

static void cmd_print(arg)
char *arg;
{
/*     FILE *pclose(); */

  char buf[LINELEN], kind;
  long start, end, i = 1;
  FILE *fp;

  clearinpline(compactmode);

  if (!internal)
    internal = 1;

  if (!*arg) {
      smart_print(var[NUMVAR].str ? var[NUMVAR].str : "");
      return;
  }

  arg = redirect(arg, buf, &kind, "print", 1, &start, &end);
  if (error || !arg) return;

  if(kind) {
#ifdef WIN32
      fp = fopen(arg, "r");
#else
      fp = (kind == '!') ? popen(arg, "r") : fopen(arg, "r");
#endif
      if(!fp) {
          PRINT_INT ("#print: #error opening %s\n", arg);
	  error = SYNTAX_ERROR;
          return;
      }
      while(!error && (!start || i <= end) && fgets(buf, LINELEN, fp))
	  if (!start || i++>=start)
	    tty_printf("%s", buf);
      tty_putc('\n');
#ifdef WIN32
		fclose(fp);
#else
      if(kind == '!') pclose(fp); else fclose(fp);
#endif
  } else
    smart_print(arg);
}

static void cmd_var(arg)
char *arg;
{
/*     FILE *pclose(); */

  char buf[LINELEN], *tmp, *tmp2, kind, type, right = 0, deleting = 0;
  varnode **ptr_named_var = NULL, *named_var = NULL;
  FILE *fp;
  long start, end, i = 1;
  int len, index;

  arg = skipspace(arg);
  tmp = first_regular(arg, '=');

  if (*tmp) {
    *tmp++ = '\0';     /* skip the = */
    right = 1;
    if (!*tmp)
      deleting = 1;
    else if (*tmp == ' ')
      tmp++;
  }

  if (*arg == '$')
    type = TYPE_TXT_VAR;
  else if (*arg == '@')
    type = TYPE_NUM_VAR;
  else {
    PRINT_INT ("#var: #error: missing variable\n");
    error = SYNTAX_ERROR;
    return;
  }
  kind = *++arg;
  if (isalpha(kind) || kind == '_') {
    kind = type==TYPE_TXT_VAR ? 1 : 0;
    ptr_named_var = lookup_varnode(arg, kind);
    if (!*ptr_named_var)
      if (!deleting) {
	named_var = add_varnode(arg, kind);
	if (error)
	  return;
	if (echo_int)
	  PRINT_INT ("#new variable: %s\n", arg - 1);
      }
      else {
	print_error(error=UNDEFINED_VARIABLE_ERROR);
	return;
      }
    else
      named_var = *ptr_named_var;

    index = named_var->index;
  }
  else {
    tmp2 = buf;
    index = xeval(&tmp2, &arg, PRINT_AS_NUM, 0);
    if (error) return;
    if (index != TYPE_NUM) {
      print_error(error=NO_NUM_VALUE_ERROR);
      return;
    }
    index = *(long *)buf;
    if (index < -NUMVAR || index >= NUMPARAM) {
      print_error(error=OUT_RANGE_ERROR);
      return;
    }
    if (type == TYPE_TXT_VAR && !deleting && !var[index+NUMVAR].str) {
      if (tmp2 = malloc(PARAMLEN))
	var[index+NUMVAR].str = tmp2;
      else
	syserr("malloc");
    }
  }

  if (deleting) {
    if (*ptr_named_var) {
      delete_varnode(ptr_named_var, (int)kind);
      if (echo_int)
	PRINT_INT ("#deleted variable: %s\n", arg - 1);
    }
    else if (type = TYPE_TXT_VAR) {
      if (tmp2 = var[index+NUMPARAM].str)
	if (index < 0)
	  free (tmp2);
	else
	  *tmp2 = '\0';
    }
    else
      var[index+NUMPARAM].num = 0;

    return;
  }
  else if (!right) {
    tmp = buf;
    *tmp++ = '#';
    *tmp++ = '(';
    *tmp++ = type == TYPE_TXT_VAR ? '$' : '@';
    if (named_var)
      sprintf(tmp, "%s=", named_var->name);
    else
      sprintf(tmp, "%d=", index);
    while (*tmp)
      tmp++;
    if (type == TYPE_TXT_VAR) {
      *tmp++ = '\"';
      escape_specials(var[index+NUMVAR].str, tmp);
      while (*tmp)
	tmp++;
      *tmp++ = '\"';
    }
    else {
      sprintf(tmp, "%ld", *var[index+NUMVAR].num);
      while (*tmp)
	tmp++;
    }
    *tmp++ = ')';
    *tmp = '\0';
    to_input_line(buf);
    return;
  }

  arg = redirect(tmp, buf, &kind, "var", 1, &start, &end);
  if (error || !arg) return;

  if(kind) {
#ifdef WIN32
	  fp = fopen(arg, "r");
#else
      fp = (kind == '!') ? popen(arg, "r") : fopen(arg, "r");
#endif
      if(!fp) {
          PRINT_INT ("#var: #error opening %s\n", arg);
	  error = SYNTAX_ERROR;
          return;
      }
      len = 0;
      i = 1;
      while(!error && (!start || i<=end) && fgets(buf+len, LINELEN-len, fp))
	 if (!start || i++>=start)
	    len += strlen(buf + len);
#ifdef WIN32
	  fclose(fp);
#else
      if(kind == '!') pclose(fp); else fclose(fp);
#endif
      if (len>PARAMLEN)
	len = PARAMLEN;
      buf[len] = '\0';
      arg = buf;
  }

  if (type == TYPE_NUM_VAR) {
    arg = skipspace(arg);
    type = 1;
    len = 0;

    if (*arg == '-')
      arg++, type = -1;
    else if (*arg == '+')
      arg++;

    if (isdigit(kind=*arg)) while (isdigit(kind)) {
      len*=10;
      len+=(kind-'0');
      kind=*++arg;
    }
    else {
      PRINT_INT ("#var: ");
      print_error (error=NO_NUM_VALUE_ERROR);
      return;
    }
    *var[index+NUMVAR].num = len * type;
  }
  else
    strcpy(var[index+NUMVAR].str, arg);
}

static void cmd_send(arg)
char *arg;
{
/*     FILE *pclose(); */

  char buf[LINELEN], kind;
  long start, end, i = 1;
  FILE *fp;

  arg = redirect(arg, buf, &kind, "send", 0, &start, &end);
  if (error || !arg) return;

  if(kind) {
#ifdef WIN32
	  fp = fopen(arg, "r");
#else
      fp = (kind == '!') ? popen(arg, "r") : fopen(arg, "r");
#endif
      if(!fp) {
          PRINT_INT ("#send: #error opening %s\n", arg);
	  error = SYNTAX_ERROR;
          return;
      }
      while(!error && (!start || i<=end) && fgets(buf, LINELEN, fp))
	  if (!start || i++>=start) {
	      buf[strlen(buf)-1] = '\0';
	      if (echo_ext) {
	          PRINT_INT ("[%s]\n", buf);
	      }
	      send_to_host(buf);
	  }
#ifdef WIN32
	  fclose(fp);
#else
      if(kind == '!') pclose(fp); else fclose(fp);
#endif
  } else {
      if (echo_ext) {
	  PRINT_INT ("[%s]\n", arg);
      }
      send_to_host(arg);
  }
}

static void cmd_settimer(arg)
char *arg;
{
  char buf[LINELEN], *temp = buf;
  int type;
  long millisec;

  arg = skipspace(arg);
  if (*arg == '(') {
    arg++;
    type = xeval(&temp, &arg, PRINT_AS_NUM, 0);
    if (error) return;

    if (*arg == ')') {
      if (type != TYPE_NUM) {
	PRINT_INT ("#settimer: ");
	print_error(error=NO_NUM_VALUE_ERROR);
      }
      else {
	vtime t;

	millisec = - *(long *)buf;
	t.tv_usec = (millisec % mSEC_PER_SEC) * uSEC_PER_mSEC;
	t.tv_sec  =  millisec / mSEC_PER_SEC;
	ref_time.tv_usec = now.tv_usec;
	ref_time.tv_sec  = now.tv_sec;
	add_vtime(&ref_time, &t);
      }
    } else {
      PRINT_INT ("#settimer: ");
      print_error(error=MISSING_PAREN_ERROR);
      return;
    }
  } else {
    PRINT_INT ("#settimer: ");
    print_error(error=MISMATCH_PAREN_ERROR);
  }
}

static void cmd_exe(arg)
char *arg;
{
/*     FILE *pclose(); */

  char buf[LINELEN], kind;
  long start, end, i = 1;
  FILE *fp;

  arg = redirect(arg, buf, &kind, "exe", 0, &start, &end);
  if (error || !arg) return;

  if(kind) {
#ifdef WIN32
	  fp = fopen(arg, "r");
#else
      fp = (kind == '!') ? popen(arg, "r") : fopen(arg, "r");
#endif
      if(!fp) {
          PRINT_INT ("#exe: #error opening %s\n", arg);
	  error = SYNTAX_ERROR;
          return;
	}
      while(!error && (!start || i<=end) && fgets(buf, LINELEN, fp))
	  if (!start || i++>=start) {
	      buf[strlen(buf)-1] = '\0';
	      parse_user_input(buf, 0);
	  }

#ifdef WIN32
	  fclose(fp);
#else
      if(kind == '!') pclose(fp); else fclose(fp);
#endif
  } else
      parse_user_input(arg, 0);
}


static void cmd_emulate(arg)
char *arg;
{
/*     FILE *pclose(); */

  char buf[LINELEN], kind;
  FILE *fp;
  long start, end, i = 1;
  int len;

  arg = redirect(arg, buf, &kind, "emulate", 0, &start, &end);
  if (error || !arg) return;

  if(kind) {
      fp = (kind == '!') ? popen(arg, "r") : fopen(arg, "r");
      if(!fp) {
          PRINT_INT ("#emulate: #error opening %s\n", arg);
	  error = SYNTAX_ERROR;
          return;
      }
      internal = -2;
      col0 = promptlen = promptstr[0] = promptintercept = '\0';

      while(!error && (!start || i<=end) && fgets(buf, LINELEN, fp))
	  if (!start || i++>=start)
	      process_remote_input(buf, strlen(buf));

      if(kind == '!') pclose(fp); else fclose(fp);
  } else {
      internal = -2;
      arg[len = strlen(arg)] = '\n';
      arg[++len] = '\0';
      col0 = promptlen = promptstr[0] = promptintercept = '\0';
      process_remote_input(arg, len);
  }
}

static void cmd_xeval(arg)
char *arg;
{
  arg = skipspace(arg);
  if (*arg=='(') {
    arg++;
    (void)xeval(NULL, &arg, PRINT_NOTHING, 0);
    if (*arg != ')') {
      PRINT_INT ("#(): ");
      print_error(error=MISSING_PAREN_ERROR);
    }
  }
  else {
    PRINT_INT ("#(): ");
    print_error(error=MISMATCH_PAREN_ERROR);
  }
}

static void cmd_action(arg)
char *arg;
{
    if(!*arg)
        show_actions();
    else
        parse_action(arg, 0);
}

static void cmd_capture(arg)
char *arg;
{
    arg = skipspace(arg);

    if(!*arg) {
        if(capfile) {
            fclose(capfile);
            capfile = NULL;
            if (echo_int) {
               PRINT_INT ("#end of capture to file.\n");
            }
        } else {
            PRINT_INT ("#capture to what file?\n");
        }
    } else {
        if(capfile) {
            PRINT_INT ("#capture already active.\n");
        } else {
            if((capfile = fopen(arg, "w")) == NULL) {
                PRINT_INT ("#error writing file %s\n", arg);
            } else if (echo_int) {
                PRINT_INT ("#capture to %s active, '#capture' ends.\n", arg);
            }
        }
    }
}

static void cmd_record(arg)
char *arg;
{
    arg = skipspace(arg);

    if(!*arg) {
        if(recordfile) {
            fclose(recordfile);
            recordfile = NULL;
            if (echo_int) {
               PRINT_INT ("#end of record to file.\n");
            }
        } else {
            PRINT_INT ("#record to what file?\n");
        }
    } else {
        if(capfile) {
            PRINT_INT ("#record already active.\n");
        } else {
            if((recordfile = fopen(arg, "w")) == NULL) {
                PRINT_INT ("#error writing file %s\n", arg);
            } else if (echo_int) {
                PRINT_INT ("#record to %s active, '#record' ends.\n", arg);
            }
        }
    }
}

static void cmd_prefix(arg)
char *arg;
{
    strcpy(prefixstr, arg);
    if (echo_int) {
       PRINT_INT ("#prefix %s.\n", *arg ? "set" : "cleared");
    }
}

#ifndef NOSHELL
static void cmd_shell(arg)
char *arg;
{
    int system();
    if(!*arg) {
        if (echo_int) {
           PRINT_INT ("#that's easy\n");
        }
    } else {
        reset_terminal();
        system(arg);
        set_terminal();
    }
    line0 = lines - 1;
    tty_gotoxy(0, line0); tty_putc('\n');
}
#endif

static void cmd_identify(arg)
char *arg;
{
    char id_string[10];
    
    strcpy(id_string, "~$#EI");

    edit_start[0] = edit_end[0] = '\0';
    if (*arg) {
        char *p = strchr(arg, ' ');
        if(p) {
            *(p++) = '\0';
            strcpy(edit_end, p);
        }
        strcpy(edit_start, arg);
    }
    send_to_host(id_string);
    if (echo_int) {
        PRINT_INT ("#identify: %s done!\n", id_string);
    }
}

/*ARGSUSED*/
static void cmd_edit(arg)
char *arg;
{
    editsess *sp;

    if(edit_sess) {
        for(sp = edit_sess; sp; sp = sp->next)
            tty_printf("# %s (%d)\n", sp->descr, sp->key);
    } else
        tty_printf("#no active editors.\n");
    if (!internal) internal = 1;
}

static void cmd_cancel(arg)
char *arg;
{
    editsess *sp;

    if(!edit_sess) {
        PRINT_INT ("#no active sessions to cancel.\n");
    } else {
        if(*arg) {
            for(sp = edit_sess; sp; sp = sp->next)
                if(atoi(arg) == sp->key) {
                    cancel_edit(sp);
                    break;
                }
            if(!sp) {
                PRINT_INT ("#no such session - use the session number\n");
            }
        } else {
            if(edit_sess->next) {
                PRINT_INT ("#several sessions active, use #cancel <number>\n");
            } else
                cancel_edit(edit_sess);
        }
    }
}

static void cmd_help(arg)
char *arg;
{
    int i, size;
    char *text, *tmp;

    arg = skipspace(arg);
    if (*arg == '#') arg++;
    if (!*arg) {
        for(size = 50, i = 0; commands[i].name; i++)
	   size += strlen(commands[i].name) + strlen(commands[i].help) + 5;

	text = tmp = (char*)malloc(size);
	if (!text) syserr("malloc");

        sprintf (tmp, "#help\n\r#commands available:\n\r");
	tmp += strlen(tmp);

        for(i = 0; commands[i].name; i++) {
	   sprintf(tmp, "#%s %s\n\r", commands[i].name, commands[i].help);
	   tmp += strlen(tmp);
	}

	if (!internal) internal = 1;
	message_edit(text, strlen(text), 1, 1);
    } else {
	FILE *f = fopen(helpfile, "r");
	char line[LINELEN];
	int len;

	if(!f) {
	    PRINT_INT ("#Can't open help file %s\n", helpfile);
	} else {
	    while((tmp = fgets(line, LINELEN, f)) &&
	          (line[0] != '@' || strncmp(line + 1, arg, strlen(arg))))
	        ;
	    if(tmp) {
	        if (!internal) internal = 1;
		if (!(text = (char*)malloc(size = LINELEN)))
		    syserr("malloc");

		strcpy(text, line); /* safe, line is not longer than LINELEN */
		i = strlen(line);

		while(fgets(line, LINELEN, f) && line[0] == '@')
		    ;     /* allow multiple commands to share the same help */

		do {
		  if ((len = strlen(line)) > size - i)
		          /* Not enough space in current buffer */

		    if (!(tmp = (char*)malloc(size += LINELEN)))
		      syserr("malloc");
		    else {
		      strcpy(tmp, text);
		      free(text);
		      text = tmp;
		    }
		   
		  sprintf(text + i, "%s\r", line);
		  //strcpy(text + i, line);
		  i += len + 1;

		} while(fgets(line, LINELEN, f) && line[0] != '@');

		message_edit(text, strlen(text), 1, 1);
	    } else {
		PRINT_INT ("#No entry for '%s' in the help file.\n", arg);
	    }
	}
    }
}

/*ARGSUSED*/
static void cmd_time(arg)
char *arg;
{
  struct tm *s;
  char buf[LINELEN];

  s = localtime((time_t *)&now.tv_sec);
  (void)strftime(buf, LINELEN - 1, "%a,  %d %b %Y  %H:%M:%S", s);
  PRINT_INT ("#current time is %s\n", buf);
}

void show_delaynode(p, in_or_at)
delaynode *p; char in_or_at;
{
  long d = diff_vtime(&p->when, &now);
  struct tm *s;
  char buf[LINELEN];

  s = localtime((time_t *)&p->when.tv_sec); 
                                   /* s now points to a calendar struct */
  if (in_or_at) {
    char buf2[LINELEN];

    if (in_or_at == 2) {
      (void)strftime(buf, LINELEN - 1, "%H%M%S", s); /* writes time in buf */
      sprintf(buf2, "#at %s (%s) %s", p->name, buf, p->command);
    }
    else
      sprintf(buf2, "#in %s (%ld) %s", p->name, d, p->command);

    to_input_line(buf2);
  }
  else {
    (void)strftime(buf, LINELEN - 1, "%H:%M:%S", s);
    PRINT_INT ("#at (%s) #in (%ld) '%s' %s\n", buf, d, p->name, p->command);
  }
}


static void show_delays()
{
   delaynode *p;
   int n = (delays ? delays->next ? 2 : 1 : 0) +
     (dead_delays ? dead_delays->next ? 2 : 1 : 0);

   PRINT_INT ("#%s delay label%s defined%c\n", n ? "The following" : "No",
	  n == 1 ? " is" : "s are", n ? ':' : '.');
   for(p = delays; p; p = p->next)
       show_delaynode(p, 0);
   for(p = dead_delays; p; p = p->next)
       show_delaynode(p, 0);
}


static void change_delaynode(p, command, millisec)
delaynode **p; char *command; long millisec;
{
  delaynode *m=*p;
  *p = m->next;
  m->when.tv_usec = (millisec % mSEC_PER_SEC) * uSEC_PER_mSEC;
  m->when.tv_sec  =  millisec / mSEC_PER_SEC;
  add_vtime(&m->when, &now);
  if (*command)
    if (strlen(command) > strlen(m->command)) {
      free((void*)m->command);
      m->command = strdup(command);
      if (!m->command) syserr("strdup");
    }
    else
      strcpy(m->command, command);
  if (millisec < 0)
    add_node((defnode*)m, (defnode**)&dead_delays, rev_time_sort);
  else
    add_node((defnode*)m, (defnode**)&delays, time_sort);
  if (echo_int) {
    PRINT_INT ("#changed ");
    show_delaynode(m, 0);
  }
}

static void new_delaynode(name, command, millisec)
char *name, *command; long millisec;
{
  vtime t;
  t.tv_usec = (millisec % mSEC_PER_SEC) * uSEC_PER_mSEC;
  t.tv_sec  =  millisec / mSEC_PER_SEC;
  add_vtime(&t, &now);
  add_delaynode(name, command, &t, millisec < 0);
}


static void cmd_in(arg)
char *arg;
{
  char buf[LINELEN], *buf2 = buf, *name;
  long millisec;
  int type;
  delaynode **p;

  arg = skipspace(arg);
  if (!*arg) {
    show_delays();
    return;
  }

  arg = first_regular(name = arg, ' ');
  if (*arg)
    *arg++ = 0;

  unescape(name);
  
  p = lookup_delay(name, 0);
  if (!*p)  p = lookup_delay(name, 1);

  if (!*arg && !*p) {
    PRINT_INT ("#unknown delay label, cannot show: %s\n", name);
    return;
  }
  if (!*arg) {
    show_delaynode(*p, 1);
    return;
  }
  if (*arg != '(') {
    PRINT_INT ("#in: ");
    print_error(error=MISMATCH_PAREN_ERROR);
    return;
  }
  arg++;    /* skip the '(' */

  type = xeval(&buf2, &arg, PRINT_AS_NUM, 0);
  if (error) return;
  if (type!=TYPE_NUM) {
    PRINT_INT ("#in :");
    print_error(error=NO_NUM_VALUE_ERROR);
    return;
  }

  if (*arg == ')') {          /* skip the ')' */
    if (*++arg == ' ')
      arg++;
  }
  else {
    PRINT_INT ("#in: ");
    print_error(error=MISSING_PAREN_ERROR);
    return;
  }

  arg = skipspace(arg);
  millisec = *(long *)buf;
  if (*p && millisec)
    change_delaynode(p, arg, millisec);
  else if (!*p && millisec)
    if (*arg)
      new_delaynode(name, arg, millisec);
    else {
      PRINT_INT ("#cannot create delay label without a command\n");
    }
  else if (*p && !millisec) {
    if (echo_int) {
        PRINT_INT ("#deleting delay label: %s %s\n", name, (*p)->command);
    }
    delete_delaynode(p);
  }
  else {
    PRINT_INT ("#unknown delay label, cannot delete: %s\n", name);
  }
}

static void cmd_at(arg)
char *arg;
{
  char buf[LINELEN], *buf2 = buf, *name;
  char dayflag=0;
  struct tm *twhen;
  int num, hour, minute, second;
  delaynode **p;
  long millisec;

  arg = skipspace(arg);
  if (!*arg) {
    show_delays();
    return;
  }

  arg = first_regular(name = arg, ' ');
  if (*arg)
    *arg++ = 0;

  unescape(name);

  p = lookup_delay(name, 0);
  if (!*p)  p = lookup_delay(name, 1);

  if (!*arg && !*p) {
    PRINT_INT ("#unknown delay label, cannot show: %s\n", name);
    return;
  }
  if (!*arg) {
    show_delaynode(*p, 2);
    return;
  }
  if (*arg != '(') {
    PRINT_INT ("#in: ");
    print_error(error=MISMATCH_PAREN_ERROR);
    return;
  }
  arg++;    /* skip the '(' */

  (void)xeval(&buf2, &arg, PRINT_AS_TEXT, 0);
  if (error) return;

  if (*arg == ')') {        /* skip the ')' */
    if (*++arg == ' ')
      arg++;
  }
  else {
    PRINT_INT ("#at: ");
    print_error(error=MISSING_PAREN_ERROR);
    return;
  }

  arg = skipspace(arg);
  twhen = localtime((time_t *)&now.tv_sec); 
                          /* put current year, month, day in calendar struct */

  buf2 = skipspace(buf);  /* convert time-string into hour, minute, second */
  if (!*buf2 || !isdigit(*buf2)) {
    PRINT_INT ("#at: ");
    print_error(error=NO_NUM_VALUE_ERROR);
    return;
  }
  num = atoi(buf2);
  second = num % 100;
  minute = (num / 100) % 100;
  hour   = num / 10000;
  if (second>59 || minute>59 || hour>23) {
    PRINT_INT ("#at: #error: invalid time.\n");
    return;
  }
  if (hour<twhen->tm_hour || hour==twhen->tm_hour && 
      (minute<twhen->tm_min || minute==twhen->tm_min &&
       second<=twhen->tm_sec))
    dayflag = 1; /* it is NOT possible to define an #at refering to the past */
                 /* if you use a time smaller than the current, it refers to
                     tomorrow */

  millisec = (hour - twhen->tm_hour) * 3600 + (minute - twhen->tm_min) * 60 + 
      second - twhen->tm_sec + (dayflag ? 24*60*60 : 0);
  millisec *= mSEC_PER_SEC; /* Comparing time with current calendar,
                                we finally got the delay */
  millisec -= now.tv_usec / uSEC_PER_mSEC;

  if (*p)
    change_delaynode(p, arg, millisec);
  else
    if (*arg)
      new_delaynode(name, arg, millisec);
    else {
      PRINT_INT ("#cannot create delay label without a command\n");
    }
}

static void cmd_do(arg)
char *arg;
{
  char buf[LINELEN], *tmp = buf;
  int type;
  long result;

  arg = skipspace(arg);
  if (*arg != '(') {
    PRINT_INT ("#do: ");
    print_error(error=MISMATCH_PAREN_ERROR);
    return;
  }
  arg++;

  type = xeval(&tmp, &arg, PRINT_AS_NUM, 0);
  if (error) return;

  if (type != TYPE_NUM) {
    PRINT_INT ("#do: ");
    print_error(error=NO_NUM_VALUE_ERROR);
    return;
  }

  if (*arg == ')') {          /* skip the ')' */
    if (*++arg == ' ')
      arg++;
  }
  else {
    PRINT_INT ("#do: ");
    print_error(error=MISSING_PAREN_ERROR);
    return;
  }

  result = *(long *)buf;
  if (tmp!=buf && result > 0)
     while (!error && result--)
       (void)parse_instruction(arg, 1, 0);
  else {
    PRINT_INT ("#do: bogus repeat count\n");
  }
}

static void cmd_if(arg)
char *arg;
{
  char buf[LINELEN], *tmp=buf;
  int type;

  arg = skipspace(arg);
  if (*arg!='(') {
    PRINT_INT ("#if: ");
    print_error(error=MISMATCH_PAREN_ERROR);
    return;
  }
  arg++;  /* skip the '(' */

  type = xeval(&tmp, &arg, PRINT_AS_NUM, 0);
  if (error) return;

  if (type!=TYPE_NUM) {
    PRINT_INT ("#if: ");
    print_error(error=NO_NUM_VALUE_ERROR);
    return;
  }

  if (*arg == ')')  {              /* skip the ')' */
    if (*++arg == ' ')
      arg++;
  }
  else {
    PRINT_INT ("#if: ");
    print_error(error=MISSING_PAREN_ERROR);
    return;
  }

  if (tmp!=buf && *(long *)buf)
    (void)parse_instruction(arg, 0, 0);
  else {
    arg = get_next_instr(arg);
    if (!strncmp(arg = skipspace(arg), "#else ", 6))
      (void)parse_instruction(arg + 6, 0, 0);
  }
}

static void cmd_for(arg)
char *arg;
{
    int type, loop=max_loop;
    char buf[LINELEN], *tmp;
    char *check, *increm = 0;
    params param;

    arg = skipspace(arg);
    if (*arg != '(') {
       PRINT_INT ("#for: ");
       print_error(error=MISMATCH_PAREN_ERROR);
       return;
    }
    set_param(&param);

    arg = skipspace(arg + 1);    /* skip the '(' */
    if (*arg != CMDSEP)
      (void)xeval(NULL, &arg, PRINT_NOTHING, 0);       /* execute <init> */

    if (*arg == CMDSEP)
      check = arg + 1;

    if (error)
      ;
    else if (*arg != CMDSEP) {
       PRINT_INT ("#for: ");
       print_error(error=MISSING_SEPARATOR_ERROR);
    }
    else while (!error && loop &&
         (*(tmp=buf) = '\0', increm=check,
         (type = xeval(&tmp, &increm, PRINT_AS_NUM, 0)) == TYPE_NUM &&
	 !error && *increm == CMDSEP && *(long *)buf)) {

      tmp = first_regular(increm + 1, ')');
      if (*tmp)
	  (void)parse_instruction(tmp + 1, 1, 1);
      else {
	  PRINT_INT ("#for: ");
	  print_error(error=MISSING_PAREN_ERROR);
      }

      if (!error) {
	  tmp = increm + 1;
	  if (*tmp != ')')
	     (void)xeval(NULL, &tmp, PRINT_NOTHING, 0);
      }

      loop--;
    }
    if (error)
      ;
    else if (increm && *increm != CMDSEP) {
       PRINT_INT ("#for: ");
       print_error(error=MISSING_SEPARATOR_ERROR);
    }
    else if (!loop) {
       PRINT_INT ("#for: ");
       print_error(error=MAX_LOOP_ERROR);
    }
    else if (type != TYPE_NUM) {
       PRINT_INT ("#for: ");
       print_error(error=NO_NUM_VALUE_ERROR);
    }
    if (error!=DYN_STACK_UND_ERROR && error!=DYN_STACK_OV_ERROR)
      restore_param();
}

static void cmd_while(arg)
char *arg;
{
    int type, loop=max_loop;
    char buf[LINELEN], *tmp;
    char *check;
    params param;

    arg = skipspace(arg);
    if (!*arg) {
       PRINT_INT ("#while: ");
       print_error(error=MISMATCH_PAREN_ERROR);
       return;
    }
    set_param(&param);

    check = ++arg;   /* skip the '(' */
    while (!error && loop &&
         (*(tmp=buf) = '\0', arg=check, 
         (type = xeval(&tmp, &arg, PRINT_AS_NUM, 0)) == TYPE_NUM &&
         !error && *arg == ')' && *(long *)buf)) {

      if (*(tmp = arg + 1) == ' ')          /* skip the ')' */
	tmp++;
      if (*tmp)
        (void)parse_instruction(tmp, 1, 1);
      loop--;
    }
    if (error)
       ;
    else if (*arg != ')') {
       PRINT_INT ("#while: ");
       print_error(error=MISSING_PAREN_ERROR);
    }
    else if (!loop) {
       PRINT_INT ("#while: ");
       print_error(error=MAX_LOOP_ERROR);
    }
    else if (type != TYPE_NUM) {
       PRINT_INT ("#while: ");
       print_error(error=NO_NUM_VALUE_ERROR);
    }
    if (error!=DYN_STACK_UND_ERROR && error!=DYN_STACK_OV_ERROR)
      restore_param();
}

void show_stat() {
  cmd_net(NULL);
  cmd_cpu(NULL);
}

/*ARGSUSED*/
static void cmd_net(arg)
char *arg;
{
  PRINT_INT ("#received from host: %ld bytes, sent to host: %ld bytes.\n",
	     received, sent);
}


#ifndef CLOCKS_PER_SEC
#define CLOCKS_PER_SEC uSEC_PER_SEC
#endif
                   /* hope it works.... */

/*ARGSUSED*/
static void cmd_cpu(arg)
char *arg;
{
    float f = (float)(cpu_clock - start_clock) / CLOCKS_PER_SEC;
    long l = diff_vtime(&now, &start_time);
    PRINT_INT ("#CPU time used: %.3f sec. (%.1f%%)\n", f,
	 l ? f * (100 * mSEC_PER_SEC) / l: (float)0);
}


#ifdef TELNETBUG
static void cmd_colour(arg)
char *arg;
{
    int attrcode;

    arg = skipspace(arg);
    if (!*arg) {
        strcpy(modenorm, backupmodenorm);
	tty_printf("%s", modenorm);	
	if (echo_int) {
	    PRINT_INT ("#standard colour cleared.\n");
	}
	tty_flush();
	return;
    }

    attrcode = parse_attributes(arg);
    if(attrcode < 0) {
        PRINT_INT ("#invalid attribute syntax\n");
	if (echo_int) show_attr_syntax();
    } else {
        int bg = BACKGROUND(attrcode), fg = FOREGROUND(attrcode);
        if(fg >=COLOURS || bg >= COLOURS) {
	    PRINT_INT ("#please specify foreground and background colours\n");
	} else {
	    sprintf(modenorm, "\033[;%c%d;%s%dm",
		    fg<LOWCOLOURS ? '3' : '9', fg % LOWCOLOURS,
		    bg<LOWCOLOURS ? "4" : "10", bg % LOWCOLOURS);
	    tty_printf("%s", modenorm);
	    if (echo_int) {
	        PRINT_INT ("#standard colour set.\n");
	    }
	    tty_flush();
        }
    }
}
#endif

static void cmd_lines(arg)
char *arg;
{
    int i=0;

    if (*arg && (i = atoi(arg)) > 0)
        lines = i;
    if (echo_int || !i) {
       PRINT_INT ("#lines: %d\n", lines);
    }
}

static void cmd_connect(arg)
char *arg;
{
#ifdef TERM
  PRINT_INT ("#connect: multiple sessions not supported in term version.\n");
#else
  char *n, *s, *s1, *s2;
  int argc = 0;

  if (!*skipspace(arg))
    show_connections();
  else
   {
     n = strtok(arg, " ");
     s = strtok(0, " ");
     if (s && *s) {
       argc = 2;
       s1 = strtok(0, " ");
       if (s1 && *s1) {
         argc++;
         s2 = strtok(0, "\0");
         if (s2 && *s2)
           argc++;
       }
     }
     if (argc <= 2) {
       if (*hostname)
         connect_sess(n, s, hostname, atoi(portnumber));
       else {
	 PRINT_INT ("#default host not defined. Please specify an address\n");
       }
     }
     else {
       if (!*hostname && strcmp(n, "main")) {
	 PRINT_INT ("#main session must be opened before subsidiary ones\n");
       }
       else if (argc == 3) {
	 if (!*hostname) {
           strcpy(hostname, s);
	   strcpy(portnumber, s1);
	 }
         connect_sess(n, NULL, s, atoi(s1));
       } else {
	 if (!*hostname) {
           strcpy(hostname, s1);
	   strcpy(portnumber, s2);
	 }
	 connect_sess(n, s, s1, atoi(s2));
       }
     }
   }
#endif /* term */
}

static void cmd_zap(arg)
char *arg;
{
  if (!*arg) {
     PRINT_INT ("#zap: no session name\n");
  }
  else
     disconnect_sess(arg);
}

static void cmd_file(arg)
char *arg;
{
  FILE *f;

  arg = skipspace(arg);
  if (*arg) {
    if (f = fopen(arg, "r"))  {
      fclose(f);
      strcpy(deffile, arg);
      if (echo_int) {
        PRINT_INT ("#save-file is now: %s\n", deffile);
      }
    }
    else {
      PRINT_INT ("#cannot find file: %s\n", arg);
    }
  }
  else if (*deffile) {
    PRINT_INT ("#current save-file: %s\n", deffile);
  }
  else {
    PRINT_INT ("#save-file not defined.\n");
  }
}


/*ARGSUSED*/
static void cmd_save(arg)
char *arg;
{
  save_settings();
  if (echo_int) {
      PRINT_INT ("#settings saved to file\n");
  }
}

/*ARGSUSED*/
static void cmd_load(arg)
char *arg;
{
  read_settings();
  if (echo_int) {
      PRINT_INT ("#settings loaded from file\n");
  }
}

static void show_history(count)
int count;
{
  int i = curline;

  if (!count) count = lines - 1;
  if (count >= MAXHIST) count = MAXHIST - 1;
  i -= count;
  if (i < 0) i += MAXHIST;

  if (!internal) internal = 1;
  while (count) {
    if (hist[i])
      tty_printf("#%2d: %s\n", count, hist[i]);
    count--;
    if (++i == MAXHIST) i= 0;
  }
}

static void exe_history(count)
int count;
{
  int i = curline;
  char buf[LINELEN];

  if (count >= MAXHIST) count = MAXHIST - 1;
  i -= count;
  if (i < 0) i += MAXHIST;
  if (hist[i]) {
    strcpy(buf, hist[i]);
    parse_user_input(buf, 0);
  }
}


static void cmd_history(arg)
char *arg;
{
  char *tmp = skipspace(arg), buf[LINELEN], *buf2 = buf;
  int num = 0;

  if (history_done) {
    print_error(error=NESTED_HISTORY_ERROR);
    return;
  }
  history_done = 1;

  if (*tmp == '(') {
    tmp++;
    num = xeval(&buf2, &tmp, PRINT_AS_NUM, 0);
    if (error)
      return;
    if (num != TYPE_NUM) {
      PRINT_INT ("#history: ");
      print_error(error=NO_NUM_VALUE_ERROR);
      return;
    }
    num = (int)*(long *)buf;
  }
  else
    num = atoi(tmp);

  if (num > 0)
    exe_history(num);  
  else
    show_history(-num);
}


static void cmd_snoop(arg)
char *arg;
{
  if (!*arg) {
     PRINT_INT ("#snoop: no session name\n");
  }
  else
     snoop_sess(arg);
}

/*ARGSUSED*/
static void cmd_stop(arg)
char *arg;
{
  delaynode *dying;

  while (delays) {
    dying = delays;
    delays = dying->next;
    dying->when.tv_sec = now.tv_sec;
    dying->when.tv_usec = now.tv_usec;
    dying->next = dead_delays;
    dead_delays = dying;
  }
  if (echo_int) {
    PRINT_INT ("#all delayed labels are now disabled\n");
  }
}
/*
 * find the first valid (non-escaped)
 * char in a string
 */
char *first_valid(p, ch)
char *p, ch;
{
  int escaped = 0;

  if (ch == NIL_CHAR)
    while(*p && *p != NIL_CHAR)
      p++;
  else while (*p && ((*p != ch) || escaped)) {
    if (*p == '\\')
      escaped = 1;
    else
      escaped = 0;
    p++;
  }
  return p;
}

/*
 * find the first regular (non-escaped, non in "" () or {} )
 * char in a string
 */
char *first_regular(p, c)
char *p, c;
{
  int escaped = 0, quotes=0, paren=0, braces=0;
  p = skipspace(p);

  while (*p && ((*p != c) || escaped || quotes || paren>0 || braces>0)) {
    if (*p == NIL_CHAR) {
      escaped = 0;
      while(*p == NIL_CHAR)
        p++;
    }
    else if (*p == '\\') {
      escaped = 1;
      while(*p == '\\')
        p++;
    }
    else {
      escaped = 0;
      if (quotes) {
        if (*p == '\"')
          quotes = 0;
      }
      else if (*p == '\"')
        quotes = 1;
      else if (*p == ')')
        paren--;
      else if (*p == '(')
        paren++;
      else if (*p == '}')
        braces--;
      else if (*p == '{')
        braces++;
      p++;
    }
  }
  return p;
}

char *get_next_instr(p)
char *p;
{
  int count, is_if;
  char *sep, *ptr;

  if (!*p)
    return p;


  p = skipspace(p);

  count = is_if = !strncmp(p, "#if", 3);

  do {
    sep   = first_regular(p, CMDSEP);

    ptr = p;
    if (*ptr) do {
      if (*ptr == '#') ptr++;

      ptr = first_regular(ptr, '#');
    } while (*ptr && strncmp(ptr, "#if", 3));

    if (sep<=ptr) {
      if (*(p = sep))
        p++;
    }
    else
      if (*ptr)
        p = get_next_instr(ptr);
      else {
        print_error(error=SYNTAX_ERROR);
        return NULL;
      }
      sep = skipspace(p);    
  } while(*p && count-- && (!is_if || !strncmp(sep, "#else", 5) && 
	  *(p = sep + 5)));

  return p;
}

/*
 * parse the #alias command
 * (if fromfile is true, call came from definition file)
 */
void parse_alias(str, fromfile)
char *str; int fromfile;
{
    char *left, *right;
    aliasnode **np;
    char buf[LINELEN];

    left = str = skipspace(str);

    str = first_regular(str, '=');

    if(*str == '=') {
        *str = '\0';
        right = ++str;
        unescape(left);
        np = lookup_alias(left);
        if(!*str) {
            /* delete alias */
            if(*np) {
                if (!fromfile && echo_int) {
                    PRINT_INT ("#deleting alias: %s=%s\n", left, (*np)->subst);
                }
                delete_aliasnode(np);
            } else {
                PRINT_INT ("#unknown alias, cannot delete: %s\n", left);
            }
        } else {
            /* add/redefine alias */

         /* direct recursion is supported (alias CAN be defined by itself) */
            if(*np) {
                free((*np)->subst);
                (*np)->subst = strdup(right);
                if(!(*np)->subst) syserr("strdup");
            } else {
                add_aliasnode(left, right);
            }
            if(!fromfile && echo_int) {
                PRINT_INT ("#new alias: %s=%s\n", left, right);
            }
        }
    } else {
        /* show alias */

        *str = '\0';
        unescape(left);
        np = lookup_alias(left);
        if(*np) {
            char *buf2=buf;
            sprintf(buf, "#alias ");
            while(*buf2) buf2++;
            escape_specials(left, buf2);
            while(*buf2) buf2++;
            *buf2++ = '=';
            strcpy(buf2, (*np)->subst);
	    if (!internal)
	      internal = 1;
            to_input_line(buf);
        } else {
            PRINT_INT ("#unknown alias, cannot show: %s\n", left);
        }
    }
}

/*
 * parse the #action command
 * this function is too damn complex because of the hairy syntax. it should be
 * split up or rewritten as an fsm instead.
 * (if fromfile is true, call came from definition file)
 */
void parse_action(str, fromfile)
char *str; int fromfile;
{
    char *p, *label, *pattern, *command, buf[LINELEN], *tmp=buf;
    actionnode **np = NULL;
    char sign, assign;
    int active, type;

    sign = *(p = skipspace(str));
    if(sign == '+' || sign == '-' || sign == '<' || sign == '>' || sign == '=')
        label = str = p + 1;
    else
        sign = 0;
                        /* labeled action: */
    if(sign) {
        p = label = str;
        p = first_regular(p, ' ');
        if (*p) *p++ = '\0'; /* p points to start of pattern, or to \0 */
        unescape(label);
        np = lookup_action(label);

                        /* '<' : remove action */
        if(sign == '<') {
            if(!np || !*np) {
                PRINT_INT ("#no action, cannot delete label: %s\n", label);
            }
            else
                delete_action(np);

                        /* '>': define action */
        } else if(sign == '>') {
            active = 1;
            if ((sign = *label) == '+' || sign == '-') {
                np = lookup_action(++label);
                if (sign == '-')
                    active = 0;
            }

            p = skipspace(p);
            if (*p == '(') {
               p++;
               type = xeval(&tmp, &p, PRINT_AS_TEXT, 0);
               if (error) return;
               if (type != TYPE_TXT) {
                  PRINT_INT ("#action: ");
                  print_error(error=NO_STRING_ERROR);
                  return;
 	       }
               if (!*(pattern = buf)) {
                  PRINT_INT ("#error: pattern of #actions must be non-null\n");
                  return;
	       }
               if (*p) 
                  p = skipspace(++p);
               if (assign = *p == '=')
                  p++;
	    }
            else {
               p = first_regular(pattern = p, '=');
               if (assign = *p)
                 *p++ = '\0';
               unescape(pattern);
	    }
            command = p;

            if(assign) {
                if(np && *np) {
                    change_action(*np, pattern, command);
                    (*np)->active = active;
                } else
                    add_new_action(label, pattern, command, active, fromfile);
            }

                        /* '=': list action */
        } else if (sign == '='){
            if (np && *np) {
                char *buf2=buf;
                sprintf(buf, "#action >%c", (*np)->active ? '+' : '-');
                while(*buf2) buf2++;
                escape_specials(label, buf2);
                while(*buf2) buf2++;
                *(buf2++) = ' ';
                escape_specials((*np)->pattern, buf2);
                while(*buf2) buf2++;
                *buf2++ = '=';
                strcpy(buf2, (*np)->command);
                to_input_line(buf);
            } else {
                PRINT_INT ("#no action, cannot list label: %s\n", label);
            }

                        /* '+', '-': turn action on/off */
        } else {
            if(np && *np) {
                (*np)->active = (sign == '+');
                if (echo_int) {
                    PRINT_INT ("#action >%s %s is now o%s.\n", label,
                        (*np)->pattern, (sign == '+') ? "n" : "ff");
                }
            } else {
                PRINT_INT ("#no action, cannot turn o%s label: %s\n",
                        (sign == '+') ? "n" : "ff", label);
            }
        }

                        /* anonymous action  */
    } else {
        pattern = str;
        command = first_regular(str, '=');

        if (*command == '=') {
            *(command++) = '\0';
	    unescape(pattern);
            np = lookup_action_pattern(pattern);
            if (*command)
                if(np && *np)
                    change_action(*np, NULL, command);
                else
                    add_anonymous_action(pattern, command, fromfile);
            else if(np && *np)
                delete_action(np);
            else {
                PRINT_INT ("#no action, cannot delete pattern: %s\n",
                        pattern);
                return;
            }
        } else {
	    unescape(pattern);
            np = lookup_action_pattern(pattern);
            if (np && *np) {
                char *buf2=buf;
                strcpy(buf, "#action ");
                while (*buf2) buf2++;
                escape_specials((*np)->pattern, buf2);
                while (*buf2) buf2++;
                *buf2++ = '=';
                strcpy(buf2, (*np)->command);
                to_input_line(buf);
            } else {
                PRINT_INT ("#no action, cannot show pattern: %s\n", pattern);
            }
        }
    }
}

/*
 * delete an action node
 */
static void delete_action(nodep)
actionnode **nodep;
{
    if (echo_int) {
        PRINT_INT ("#deleting action: >%c%s %s\n", (*nodep)->active ?
	     '+' : '-', (*nodep)->label, (*nodep)->pattern);
    }
    delete_actionnode(nodep);
}

/*
 * change fields of an existing action node
 * pattern or commands can be NULL if no change
 */
static void change_action(node, pattern, command)
actionnode *node; char *pattern, *command;
{
    if(pattern) {
        free(node->pattern);
        node->pattern = strdup(pattern);
    }
    if(command) {
        free(node->command);
        node->command = strdup(command);
    }

    if (echo_int) {
        PRINT_INT ("#changed action >%c%s %s=%s\n", node->active ? '+' : '-',
           node->label, node->pattern, node->command);
    }
}

/*
 * add an action with numbered label
 */
void add_anonymous_action(pattern, command, fromfile)
char *pattern, *command; int fromfile;
{
    static last = 0;
    char label[16];
    do {
        sprintf(label, "%d", ++last);
    } while(*lookup_action(label));
    add_new_action(label, pattern, command, 1, fromfile);
}

/*
 * create new action
 */
static void add_new_action(label, pattern, command, active, fromfile)
char *label, *pattern, *command; int active, fromfile;
{
    add_actionnode(pattern, command, label, active);
    if(!fromfile && echo_int) {
        PRINT_INT ("#new action: >%c%s %s=%s\n", active ? '+' : '-', label,
                pattern, command);
    }
}

/*
 * parse arguments to the #mark command
 * (if fromfile is true, call came from definition file)
 */
void parse_marker(str, fromfile)
char *str; int fromfile;
{
    char *p;
    marknode **np;
  
    if (*str == '=') {
      PRINT_INT ("#marker must be non-null\n");
      return;
    }
    p = first_regular(str, '=');
    if(!*p) {
        unescape(str);
        np = lookup_marker(str);
        if(*np && !fromfile) {
            char buf[LINELEN], *buf2=buf;
            strcpy(buf, "#mark ");
            while(*buf2) buf2++;
            escape_specials((*np)->pattern, buf2);
            while(*buf2) buf2++;
            *buf2++ = '=';
            strcpy(buf2, attr_name((*np)->attrcode));
            to_input_line(buf);
        } else {
            PRINT_INT ("#unknown marker, cannot show: %s\n", str);
        }

    } else {
        int attrcode;

        *(p++) = '\0';
	p = skipspace(p);
        unescape(str);
        np = lookup_marker(str);
        attrcode = parse_attributes(p);
        if(attrcode < 0) {
            PRINT_INT ("#invalid attribute syntax\n");
            if (echo_int) show_attr_syntax();
        } else if(!*p)
	    if (*np) {
	        if (echo_int) {
		    PRINT_INT ("#deleting mark: %s=%s\n", (*np)->pattern,
			   attr_name((*np)->attrcode));
		}
		delete_marknode(np);
	    } else {
	        PRINT_INT ("#unknown marker, cannot delete: %s\n", str);
	    }
        else {
            if(*np) {
                (*np)->attrcode = attrcode;
                if(!fromfile && echo_int) {
                    PRINT_INT ("#changed");
                }
            } else {
                add_marknode(str, attrcode);
                if(!fromfile && echo_int) {
                    PRINT_INT ("#new");
                }
            }
            if(!fromfile && echo_int)
                tty_printf(" mark: %s=%s\n", str, attr_name(attrcode));
        }
    }
}

/*
 * show defined marks
 */
static void show_marks()
{
    marknode *p;
    PRINT_INT ("#%s marker%s defined%c\n", markers ? "The following" : "No",
           (markers && !markers->next) ? " is" : "s are",
           markers ? ':' : '.');
    for(p = markers; p; p = p->next)
        tty_printf("#mark %s=%s\n", p->pattern, attr_name(p->attrcode));
}

/*
 * show defined aliases
 */
static void show_aliases()
{
    aliasnode *p;
  
    PRINT_INT ("#%s alias%s defined%c\n", aliases ? "The following" : "No",
           (aliases && !aliases->next) ? " is" : "es are",
           aliases ? ':' : '.');
    for(p = aliases; p; p = p->next)
        tty_printf("#alias %s=%s\n", p->name, p->subst);
}

/*
 * show defined actions
 */
static void show_actions()
{
    actionnode *p;
  
    PRINT_INT ("#%s action%s defined%c\n", actions ? "The following" : "No",
           (actions && !actions->next) ? " is" : "s are", actions ? ':' : '.');
    for(p = actions; p; p = p->next)
        tty_printf("#action >%c%s %s=%s\n", p->active ? '+' : '-', p->label,
               p->pattern, p->command);
}

/*
 * add escapes to protect special characters from being escaped.
 */
void escape_specials(str, p)
char *str, *p;
{
  while(*str) {
    if (*str == '\\') {
      while (*str == '\\')
        *p++ = *str++;

      if (!*str || *str == NIL_CHAR)
        *p++ = NIL_CHAR;
      else
        *p++ = '\\';
    }      
    else if (strchr(SPECIAL_CHARS, *str))
      *p++ = '\\';

    if (*str)
      *p++ = *str++;
  }
  *p = '\0';
}

/*ARGSUSED*/
static void cmd_ver(arg)
char *arg;
{
    printver();
}

static void cmd_quote(arg)
char *arg;
{
    arg = skipspace(arg);
    if (!*arg)
      verbatim ^= 1;
    else if (!strcmp(arg, "on"))
      verbatim = 1;
    else if (!strcmp(arg, "off"))
      verbatim = 0;
    if (echo_int) {
        PRINT_INT ("#%s mode.\n", verbatim ? "verbatim" : "normal");
    }
}

static void cmd_wrap(arg)
char *arg;
{
    arg = skipspace(arg);
    if (!*arg)
      wrap ^= 1;
    else if (!strcmp(arg, "on"))
      wrap = 1;
    else if (!strcmp(arg, "off"))
      wrap = 0;
    if (echo_int) {
        PRINT_INT ("#word wrap %sabled.\n", wrap ? "en" : "dis");
    }
}

static void cmd_compact(arg)
char *arg;
{
    arg = skipspace(arg);
    if (!*arg)
      compactmode ^= 1;
    else if (!strcmp(arg, "on"))
     compactmode = 1;
    else if (!strcmp(arg, "off"))
      compactmode = 0;

    if (echo_int) {
       PRINT_INT ("#compact mode is now o%s.\n", compactmode ? "n" : "ff");
    }
}

static void cmd_echo(arg)
char *arg;
{
    arg = skipspace(arg);
    if (!*arg)
      echo_ext ^= 1;
    else if (!strcmp(arg, "on"))
      echo_ext = 1;
    else if (!strcmp(arg, "off"))
      echo_ext = 0;

    if (echo_int) {
       PRINT_INT ("#echo is now o%s.\n", echo_ext ? "n" : "ff");
    }
}

static void cmd_debug(arg)
char *arg;
{
    arg = skipspace(arg);
    if (!*arg)
      debug ^= 1;
    else if (!strcmp(arg, "on"))
      debug = 1;
    else if (!strcmp(arg, "off"))
      debug = 0;

    if (echo_int) {
       PRINT_INT ("#debug is now o%s.\n", debug ? "n" : "ff");
    }
}

static void cmd_info(arg)
char *arg;
{
    arg = skipspace(arg);
    if (!*arg)
      echo_int ^= 1;
    else if (!strcmp(arg, "on"))
      echo_int = 1;
    else if (!strcmp(arg, "off"))
      echo_int = 0;

    PRINT_INT ("#info is now o%s.\n", echo_int ? "n" : "ff");
}

static void cmd_speedwalk(arg)
char *arg;
{
    arg = skipspace(arg);
    if (!*arg)
      speedwalk ^= 1;
    else if (!strcmp(arg, "on"))
      speedwalk = 1;
    else if (!strcmp(arg, "off"))
      speedwalk = 0;
    if (echo_int) {
       PRINT_INT ("#speedwalk is now o%s.\n", speedwalk ? "n" : "ff");
    }
}
