/*
 * edit.c
 * line editing functions for cancan
 */

#include <stdio.h>
#include <stdlib.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#ifndef WIN32
#include <sys/time.h>
#endif

#include "cancan.h"
#include "edit.h"
#include "term.h"
#include "xeval.h"

void scrollup();
void scrolldown();

edit_function internal_functions[] = {
  {NULL, (void (*)()) NULL, },
  {"&enter-line", enter_line, },
  {"&complete-word", complete_word, },
  {"&complete-line", complete_line, },
  {"&del-char-left", del_char_left, },
  {"&del-char-right",  del_char_right, },
  {"&prev-char", prev_char, },
  {"&prev-line", prev_line, },
  {"&next-char", next_char, },
  {"&next-line", next_line, },
  {"&to-history", enter_into_history, },
  {"&clear-line", clear_line, },
  {"&begin-of-line", beginning_of_line, },
  {"&end-of-line", end_of_line, },
  {"&kill-to-eol", kill_to_eol, },
  {"&transpose", transpose_chars, },
  {"&redraw-line", redraw_input_line, },
  {"&suspend", suspend_cancan, },
  {"&del-word-left", del_word_left, },
  {"&del-word-right", del_word_right, },
  {"&prev-word", prev_word, },
  {"&next-word", next_word, },
  {"&scrollup", scrollup },
  {"&scrolldown", scrolldown },
  {NULL, (void (*)()) NULL }
};


int lookup_edit_name(name)
char *name;
{
  int i;
  char *fname;

  for (i=1; fname = internal_functions[i].name; i++)
    if (!strcmp(name, fname))
      return i;

  return 0;
}

int lookup_edit_function(funct)
void (*funct)();
{
  int i;
  void (*ffunct)();

  for (i=1; ffunct = internal_functions[i].funct; i++)
    if (funct == ffunct)
      return i;

  return 0;
}

  
/*
 * insert a typed character (if it is printable)
 */
void insert_char(c)
int c;
{
    char ch;
    if(edlen < LINELEN - 2 && c >= ' ') {
	ch = c;
	insertchars(&ch, 1);
	if(ISRPAREN(c)) flashparen(c);
    }
}

/*
 * redraw the input line
 */
void redraw_input_line()
{
    clearinpline(0);
    col0 = promptlen = promptstr[0] = promptintercept = '\0'; 
    tty_putc('\n');
    if (line0 < lines - 1)
      line0 ++;
    redraw_input();
}

/*
 * transpose two characters to the left
 */
void transpose_chars()
{
    int i, j;
    unsigned char c;
    if(pos > 1 || pos > 0 && pos < edlen) {
        if (pos < edlen) {
	  j = pos;
	  i = pos - 1;
	} else {
	  j = pos - 1;
	  i = pos - 2;
	}
	c = edbuf[j]; edbuf[j] = edbuf[i]; edbuf[i] = c;
	optmove(CURLINE(pos), CURCOL(pos), CURLINE(i), CURCOL(i));
	tty_putc(edbuf[i]);
	optmove(CURLINE(i+1), CURCOL(i+1), CURLINE(j), CURCOL(j));
	tty_putc(edbuf[j]);
	if(pos < edlen) {
	    pos++;
	    optmove(CURLINE(j+1), CURCOL(j+1), CURLINE(pos), CURCOL(pos));
	}
	tty_flush();
    }
}

/*
 * erase everything to the end of line
 */
void kill_to_eol()
{
    tty_puts(clreoln);
    if(CURLINE(edlen) > CURLINE(pos)) {
	tty_printf("\n%s", clreoscr);
	optmove(CURLINE(pos) + 1, 0, CURLINE(pos), CURCOL(pos));
    }
    edbuf[pos] = '\0';
    edlen = pos;
    tty_flush();
}

/*
 * move cursor to end of line
 */
void end_of_line()
{
    optmove(CURLINE(pos), CURCOL(pos), CURLINE(edlen), CURCOL(edlen));
    tty_flush();
    pos = edlen;
}

/*
 * move cursor to beginning of line
 */
void beginning_of_line()
{
    optmove(CURLINE(pos), CURCOL(pos), line0, col0);
    pos = 0;
    tty_flush();
}

/*
 * delete a character to the right
 */
void del_char_right()
{
    if(pos < edlen)
	deletechars(1);
}

/*
 * delete a character to the left
 */
void del_char_left()
{
    if(pos) {
	pos--;
	optmove(CURLINE(pos + 1), CURCOL(pos + 1), CURLINE(pos), CURCOL(pos));
	deletechars(1);
    }
}

/*
 * enter a line into history, but don't do anything else
 */
void enter_into_history()
{
    if (!edlen)
      return;
    clearinpline(0);
    put_in_history(edbuf);
    pickline = curline;
    *edbuf = '\0';
    pos = edlen = 0;
}

/*
 * clear input line, but do nothing else
 */
void clear_line()
{
    if (!edlen)
      return;
    clearinpline(0);
    pickline = curline;
    *edbuf = '\0';
    pos = edlen = 0;
}

void getsbline();

/*
 * enter a line
 */
void enter_line()
{
    char *p;
    
    optmove(CURLINE(pos), CURCOL(pos), CURLINE(edlen), CURCOL(edlen));
	tty_putc('\n');
	getsbline();

    line0 = CURLINE(edlen);
    if(line0 < lines - 1) line0++;

    if (recordfile)
        fprintf(recordfile, "%s\n", edbuf);

    if(!*edbuf || verbatim && edbuf[0] != '#') {
	send_to_host(edbuf);
	col0 = promptlen = promptstr[0] = promptintercept = '\0'; 
    } else {
        col0 = 0;
        error = internal = 0;

        parse_user_input(edbuf, 1);

	if (internal >= 0) {
            showprompt();
	    col0 = printstrlen(promptstr);
	    promptlen = strlen(promptstr);
	} else if (internal != -2)
	    col0 = promptlen = promptstr[0] = promptintercept = '\0'; 

    }
    /* don't put identical lines in history, nor short ones */
    p = hist[curline ? curline - 1 : MAXHIST - 1];
    if (!history_done && (!p || edlen > 1 && strcmp(edbuf, p)))
       put_in_history(edbuf);
    history_done = 0;
    pickline = curline;
    if(inserted_next[0]) {
	strcpy(edbuf, inserted_next);
	inserted_next[0] = '\0';
    } else
	strcpy(edbuf, prefixstr);
    pos = edlen = strlen(edbuf);
    redraw_input();
}

/*
 * move one word forward
 */
void next_word()
{
    int i;
    for(i = pos; !isalnum(edbuf[i]) && edbuf[i]; i++);
    while(isalnum(edbuf[i])) i++;
    optmove(CURLINE(pos), CURCOL(pos), CURLINE(i), CURCOL(i));
    pos = i;
    tty_flush();
}

/*
 * move one word backward
 */
void prev_word()
{
    int i;
    for(i = pos; i && !isalnum(edbuf[i - 1]); i--);
    while(i && isalnum(edbuf[i - 1])) i--;
    optmove(CURLINE(pos), CURCOL(pos), CURLINE(i), CURCOL(i));
    pos = i;
    tty_flush();
}

/*
 * delete word to the right
 */
void del_word_right()
{
    int i;
    for(i = pos; !isalnum(edbuf[i]) && i < edlen; i++);
    while(isalnum(edbuf[i])) i++;
    deletechars(i - pos);
}

/*
 * delete word to the left
 */
void del_word_left()
{
    int i;
    for(i = pos; i && !isalnum(edbuf[i - 1]); i--);
    while(i && isalnum(edbuf[i - 1])) i--;
    optmove(CURLINE(pos), CURCOL(pos), CURLINE(i), CURCOL(i));
    i = pos - i; pos -= i;
    deletechars(i);
}

/*
 * get previous line from history list
 */
void prev_line()
{
    int i = pickline - 1;
    if(i < 0) i = MAXHIST - 1;
    if(hist[i]) {
	if(hist[pickline] && strcmp(hist[pickline], edbuf)) {
	  free(hist[pickline]);
	  hist[pickline] = NULL;
	}
	if (!hist[pickline])
	  hist[pickline] = strdup(edbuf);
	pickline = i;
	clearinpline(0);
	strcpy(edbuf, hist[pickline]);
	pos = edlen = strlen(edbuf);
	redraw_input();
    }
}

/*
 * get next line from history list
 */
void next_line()
{
    int i = pickline + 1;
    if(i == MAXHIST) i = 0;
    if(hist[i]) {
	if(hist[pickline] && strcmp(hist[pickline], edbuf)) {
	  free(hist[pickline]);
	  hist[pickline] = NULL;
	}
	if (!hist[pickline])
	  hist[pickline] = strdup(edbuf);
	pickline = i;
	clearinpline(0);
	strcpy(edbuf, hist[pickline]);
	edlen = pos = strlen(edbuf);
	redraw_input();
    }
}

/*
 * move one char backward
 */
void prev_char()
{
    if(pos) {
	pos--;
	optmove(CURLINE(pos + 1), CURCOL(pos + 1), CURLINE(pos), CURCOL(pos));
	tty_flush();
    }
}

/*
 * move one char forward
 */
void next_char()
{
    if(pos < edlen) {
	pos++;
	optmove(CURLINE(pos - 1), CURCOL(pos - 1), CURLINE(pos), CURCOL(pos));
	tty_flush();
    }
}

/*
 * execute string as if typed
 */
void key_run_command(cmd)
char *cmd;
{
    clearinpline(0);
    tty_printf("%s\n", cmd);
    col0 = 0;
    error = internal = 0;
    if (recordfile)
      fprintf(recordfile, "%s\n", cmd);

    parse_instruction(cmd, 1, 0);
    if (internal >= 0)
        showprompt();
    else if (internal != -2)
        promptlen = promptstr[0] = promptintercept = '\0'; 

    history_done = 0;
    redraw_input();
}
