/*
 * list utility functions:
 * I've given up the idea of having one generic list type, so everything
 * is duplicated in functions for aliases, actions, markers and keydefs.
 * Better alternatives, anyone?
 */

#include <stdio.h>
#include <stdlib.h>
#ifndef WIN32
#include <unistd.h>
#endif
#include <sys/types.h>
#ifndef WIN32
#include <sys/time.h>
#endif
#include "cancan.h"
#include "cmd.h"
#include "xeval.h"

char *strdup();

/*
 * compare two times, return -1 if t1 < t2, 1 if t1 > t2, 0 if t1 == t2
 */
int cmp_vtime(t1, t2)
vtime *t1, *t2;
{
  int i;
  i = t1->tv_sec < t2->tv_sec ? -1 : t1->tv_sec > t2->tv_sec ? 1 : 0;
  if (!i)
    i = t1->tv_usec < t2->tv_usec ? -1 : t1->tv_usec > t2->tv_usec ? 1 : 0;
  return i;
}

void add_vtime(t1, t2)
vtime *t1, *t2;
{
  t1->tv_sec += t2->tv_sec;
  if ((t1->tv_usec += t2->tv_usec) >= uSEC_PER_SEC) {
    t1->tv_sec += t1->tv_usec / uSEC_PER_SEC;
    t1->tv_usec %= uSEC_PER_SEC;
  }
}

/*
 * Return t1 - t2, in milliseconds
 */
long diff_vtime(t1, t2)
vtime *t1, *t2;
{
  return (t1->tv_sec - t2->tv_sec) * mSEC_PER_SEC +
       (t1->tv_usec - t2->tv_usec) / uSEC_PER_mSEC;
}

/*
 * standard ASCII comparison between nodes
 */
int ascii_sort(node1, node2)
defnode *node1, *node2;
{
  return strcmp(node1->sortfield, node2->sortfield);
}

/*
 * comparison between times of execution of nodes
 * (return -1 if node1->when < node2->when)
 */
int time_sort(node1, node2)
defnode *node1, *node2;
{
  return cmp_vtime(&((delaynode *)node1)->when, &((delaynode *)node2)->when);
}

/*
 * reverse comparison between times of execution of nodes
 * (return -1 if node1->when > node2->when)
 */
int rev_time_sort(node1, node2)
defnode *node1, *node2;
{
  return -cmp_vtime(&((delaynode *)node1)->when, &((delaynode *)node2)->when);
}


/*
 * generic list node adding routine
 */
void add_node(newnode, base, sort)
defnode *newnode, **base; int (*sort)();
{
    while((*base) && (!sort || (*sort)(newnode, *base) > 0))
	base = &(*base)->next;
    newnode->next = *base;
    *base = newnode;
}


/*
 * add a node to the alias list
 */
void add_aliasnode(name, subst)
char *name, *subst;
{
    aliasnode *new = (aliasnode*)malloc(sizeof(aliasnode));
    if(!new) syserr("malloc");
    new->name = name ? strdup(name) : name;
    new->subst = subst ? strdup(subst) : subst;
    if(name && !new->name || subst && !new->subst) syserr("strdup");
    add_node((defnode*)new, (defnode**)&aliases, ascii_sort);
}
    
/*
 * add a node to the action list
 */
void add_actionnode(pattern, command, label, active)
char *pattern, *command, *label; int active;
{
    actionnode **p, *new = (actionnode*)malloc(sizeof(actionnode));
    int i;
    if(!new) syserr("malloc");
    new->pattern = strdup(pattern);
    new->command = command ? strdup(command) : command;
    new->label = strdup(label);
    new->active = active;
    if(!new->pattern || command && !new->command ||
       label && !new->label) syserr("strdup");
#ifdef SORT
    add_node((defnode*)new, (defnode**)&actions, ascii_sort);
#else
    for (p=&actions, i=1; *p && (a_nice==0 || i<a_nice); p = &(*p)->next, i++)
      ;
    new->next = *p;
    *p = new;
#endif
}

/*
 * add a node to the marker list
 */
void add_marknode(pattern, attrcode)
char *pattern; int attrcode;
{
    marknode **p, *new = (marknode*)malloc(sizeof(marknode));
    int i;
    if(!new) syserr("malloc");
    new->pattern = strdup(pattern);
    new->attrcode = attrcode;
    if(!new->pattern) syserr("strdup");
#ifdef SORT
    add_node((defnode*)new, (defnode**)&markers, ascii_sort);
#else
    for (p=&markers, i=1; *p && (a_nice==0 || i<a_nice); p = &(*p)->next, i++)
      ;
    new->next = *p;
    *p = new;
#endif
}

/*
 * add a node to the keydef list
 */
void add_keynode(name, sequence, funct, call_data)
char *name, *sequence; void (*funct)(); char *call_data;
{
    keynode *new = (keynode*)malloc(sizeof(keynode));
    if(!new) syserr("malloc");
    new->name = strdup(name);
    new->sequence = strdup(sequence);
    new->funct = funct;
    new->call_data = call_data ? strdup(call_data) : call_data;
    if(!new->name || !new->sequence || call_data && !new->call_data)
        syserr("strdup");
    add_node((defnode*)new, (defnode**)&keydefs, ascii_sort);
}

/*
 * add a node to the delayed command list
 * is_dead == 1 means when < now (and so cannot be executed anymore)
 */
void add_delaynode(name, command, when, is_dead)
char *name, *command; vtime *when; char is_dead;
{
    delaynode *new = (delaynode*)malloc(sizeof(delaynode));
    if(!new) syserr("malloc");
    new->name = strdup(name);
    new->command = command ? strdup(command) : command;
    if(!new->name || command && !new->command) syserr("strdup");
    new->when.tv_sec = when->tv_sec;
    new->when.tv_usec = when->tv_usec;
    if (is_dead)
      add_node((defnode*)new, (defnode**)&dead_delays, rev_time_sort);
    else
      add_node((defnode*)new, (defnode**)&delays, time_sort);
    if (echo_int) {
      printf("#new ");
      show_delaynode(new, 0);
    }
}

/*
 * add a node to named variables list
 */
varnode *add_varnode(name, type)
char *name; int type;
{
    varnode *new;
    int m;

    if (num_named_vars[type] >= max_named_vars) {
      if ((m = max_named_vars + NUMVAR + NUMPARAM) + 100 < 0) { /* overflow */
	print_error(error=OUT_OF_VAR_SPACE_ERROR);
	return (varnode *)0;
      }
      else {
	vars *newvar;
	int i;
	if (newvar = (vars *)malloc((m + 100) * sizeof(vars))) {
	  for (i=0; i<m; i++) {
	    newvar[i].str = var[i].str;
	    newvar[i].num = var[i].num;
	  }
	  max_named_vars += 100;
	  for (i=m, m+=100; i<m; i++) {
	    newvar[i].str = NULL;
	    newvar[i].num = NULL;
	  }
	  free((void *)var);
	  var = newvar;
	}
	else {
	  print_error(error=OUT_OF_VAR_SPACE_ERROR);
	  return (varnode *)0;
	}
      }
    }

    new = (varnode*)malloc(sizeof(varnode));
    if(!new) syserr("malloc");
    new->name = name ? strdup(name) : name;
    if(name && !new->name) syserr("strdup");

    if (type)
      if (new->str = (char *)malloc(PARAMLEN))
	*new->str = '\0';
      else
        syserr("malloc");
    else {
      new->num = 0;
      new->str = 0;
    }

    new->index = NUMPARAM + num_named_vars[type]++;

    if (type)
      var[new->index + NUMVAR].str = new->str;
    else
      var[new->index + NUMVAR].num = &new->num;

    add_node((defnode*)new, (defnode**)&named_vars[type], ascii_sort);
    return new;
}

/*
 * look up an alias node by name:
 * return pointer to pointer to node or a pointer to NULL if nothing found
 */
aliasnode **lookup_alias(name)
char *name;
{
    aliasnode **p = &aliases;
    while(*p && strcmp(name, (*p)->name))
	p = &(*p)->next;
    return p;
}

/*
 * look up an action node by label:
 * return pointer to pointer to node or a pointer to NULL if nothing found
 */
actionnode **lookup_action(label)
char *label;
{
    actionnode **p = &actions;
    while(*p && strcmp(label, (*p)->label))
	p = &(*p)->next;
    return p;
}

/*
 * look up an action node by pattern:
 * return pointer to pointer to node or a pointer to NULL if nothing found
 */
actionnode **lookup_action_pattern(pattern)
char *pattern;
{
    actionnode **p = &actions;
    while(*p && strcmp(pattern, (*p)->pattern))
	p = &(*p)->next;
    return p;
}

/*
 * look up an marker node by pattern:
 * return pointer to pointer to node or a pointer to NULL if nothing found
 */
marknode **lookup_marker(pattern)
char *pattern;
{
    marknode **p = &markers;
    while(*p && strcmp(pattern, (*p)->pattern))
	p = &(*p)->next;
    return p;
}

/*
 * look up an marker node by pattern:
 * return pointer to pointer to node or a pointer to NULL if nothing found
 */
keynode **lookup_key(name)
char *name;
{
    keynode **p = &keydefs;
    while(*p && strcmp(name, (*p)->name))
	p = &(*p)->next;
    return p;
}

/*
 * look up a delayed command node by label:
 * return pointer to pointer to node or a pointer to NULL if nothing found
 */
delaynode **lookup_delay(name, is_dead)
char *name, is_dead;
{
    delaynode **p = (is_dead ? &dead_delays : &delays);
    while(*p && strcmp(name, (*p)->name))
        p = &(*p)->next;
    return p;
}

/*
 * look up a named variable node by name:
 * return pointer to pointer to node or a pointer to NULL if nothing found
 */
varnode **lookup_varnode(name, type)
char *name; int type;
{
    varnode **p = &named_vars[type];
    while(*p && strcmp(name, (*p)->name))
	p = &(*p)->next;
    return p;
}

/*
 * delete an alias node, given a pointer to its precessor's pointer
 */
void delete_aliasnode(base)
aliasnode **base;
{
    aliasnode *p = *base;
    if (p->name) free(p->name);
    if (p->subst) free(p->subst);
    *base = p->next;
    free((void*)p);
}

/*
 * delete an action node, given a pointer to its precessor's pointer
 */
void delete_actionnode(base)
actionnode **base;
{
    actionnode *p = *base;
    if (p->pattern) free(p->pattern);
    if (p->command) free(p->command);
    if (p->label) free(p->label);
    *base = p->next;
    free((void*)p);
}

/*
 * delete an marker node, given a pointer to its precessor's pointer
 */
void delete_marknode(base)
marknode **base;
{
    marknode *p = *base;
    if (p->pattern) free(p->pattern);
    *base = p->next;
    free((void*)p);
}

/*
 * delete a keydef node, given a pointer to its precessor's pointer
 */
void delete_keynode(base)
keynode **base;
{
    keynode *p = *base;
    if (p->name) free(p->name);
    if (p->sequence) free(p->sequence);
    if (p->call_data) free(p->call_data);
    *base = p->next;
    free((void*)p);
}

/*
 * delete a delayed command node, given a pointer to its precessor's pointer
 */
void delete_delaynode(base)
delaynode **base;
{
    delaynode *p = *base;
    if (p->name) free(p->name);
    if (p->command) free(p->command);
    *base = p->next;
    free((void*)p);
}

/*
 * delete a named variable node, given a pointer to its precessor's pointer
 */
void delete_varnode(base, type)
varnode **base; int type;
{
    varnode *p = *base;
    int index = p->index, i;

    if (type) {
      var[index + NUMVAR].str = NULL;
      if (p->str)
	free(p->str);
    }
    else
      var[index + NUMVAR].num = NULL;

    if (p->name) free(p->name);
    *base = p->next;
    free((void*)p);

    i = NUMPARAM + --num_named_vars[type];

    if (index == i)
      return;

    /* now I must fill the hole in var[index].*** */

    for (p = named_vars[type]; p && p->index < i; p = p->next)
      ;

    if (!p) {               /* should NEVER happen */
      print_error(error=UNDEFINED_VARIABLE_ERROR);
      return;
    }

    p->index = index;

    if (type) {
      var[index + NUMVAR].str = p->str;
      var[    i + NUMVAR].str = NULL;
    } else {
      var[index + NUMVAR].num = &p->num;
      var[    i + NUMVAR].num = NULL;
    }
}
