/*
 * tel.c -- telnet protocol communication module for cancan
 */

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include "telnet.h"
#include <errno.h>
#define TELOPT_NAWS 31

#include "cancan.h"
#include "term.h"

#ifdef TELOPTS
#define TELOPTSTR(n) ((n) > NTELOPTS ? "unknown" : telopts[n])
#endif

void bcopy(), bzero();


#ifdef NOBCOPY
#define bcopy(a,b,c) my_memmove ((b), (a), (c))
#endif

#ifdef NOBZERO
#define bzero(a,b) memset ((a), 0, (b))
#endif

static void dosubopt();
static void sendopt();

static int sending_size = 0;	/* true if we're communicating window size */

/*
 * connect to remote host
 * Warning: some voodoo code here
 */
int doconnect(addr, port)
char *addr; int port;
{
  struct sockaddr_in address;
  struct hostent *host_info;
  int err;
  int nsock;
  
  bzero(&address, sizeof(address));
  /*
   * inet_addr has a strange design: It is documented to return -1 for
   * malformed requests, but it is declared to return unsigned long!
   * Anyway, this works.
   */    


  address.sin_addr.s_addr = inet_addr(addr);
  if(address.sin_addr.s_addr != (unsigned int)-1)
     address.sin_family = AF_INET;
  else
   {
     host_info = gethostbyname(addr);
     if(host_info == 0)
     	syserr("gethostbyname:unknown host");
     bcopy(host_info->h_addr, (char*)&address.sin_addr,
	   host_info->h_length);
     address.sin_family = host_info->h_addrtype;
   }
  address.sin_port = htons(port);
  
  nsock = socket(address.sin_family, SOCK_STREAM, 0);
  if(nsock < 0)
     syserr("creating socket");
  
  tty_printf("#Trying %s... \n", addr); fflush(stdout);
  err = connect(nsock, (struct sockaddr *)&address,
				sizeof(address));

  if(err < 0)
      switch(WSAGetLastError()) {
       case WSAECONNREFUSED:
         tty_printf("#ERROR - CONNECTION REFUSED.\n");
         break;
       case WSAENETUNREACH:
         tty_printf("#ERROR - THE NETWORK IS NOT REACHABLE FROM THIS HOST.\n");
         break;
	  }
 
  if ((err < 0) || nsock >= MAXFDSCAN) 
  { 
	  //tty_printf("Something went wrong. Closing socket. %d, %d\n", nsock, MAXFDSCAN);
		closesocket(nsock);
		return -1;
   }
  
  tty_printf("connected on fd %d \n", nsock);
  return nsock;
}

/*
 * read a maximum of maxbytes from remote host into buffer using the telnet
 * protocol. return bytes read.
 */
int read_socket(buffer, maxbytes)
char *buffer; int maxbytes;
{
    static enum {
	NORMAL, GOTIAC, GOTWILL, GOTWONT, GOTDO, GOTDONT, GOTSB, GOTSBIAC
    } state = NORMAL;
    static unsigned char buf[SOCKBUFSIZE];
    int i, n;
    unsigned char *p;
    static unsigned char subopt[MAXSUBOPT];
    int subchars;
    
    if(maxbytes > SOCKBUFSIZE) maxbytes = SOCKBUFSIZE;

	while((n = recv(sockfd, buf, maxbytes, 0)) == SOCKET_ERROR && errno == WSAEINTR);
	
	if(n == 0) {
		disconnect_sess(NULL);
		return 0;
    } else if(n < 0) {
		syserr("read from socket");
    }

    /* copy data to buffer and interpret telnet protocol escapes */

    for(i = 0, p = (unsigned char*)buffer; i < n; i++) {
	switch(state) {
	case NORMAL:
	    if(buf[i] == IAC)
		state = GOTIAC;
	    else
		*p++ = buf[i];
	    break;
	case GOTIAC:
	    switch(buf[i]) {
	    case WILL:
		state = GOTWILL; break;
	    case WONT:
		state = GOTWONT; break;
	    case DO:
		state = GOTDO; break;
	    case DONT:
		state = GOTDONT; break;
	    case SB:
		state = GOTSB;
		subchars = 0;
		break;
	    case IAC:
		*p++ = IAC;
		break;
	    case GA:
		/* I should handle GA as end-of-prompt marker one day */
	    default:
		/* ignore the rest of the telnet commands */
#ifdef TELOPTS
		tty_printf("[skipped IAC <%d>]\n", buf[i]);
		*p++ = '@';
#endif
		state = NORMAL; break;
	    }
	    break;
	
	case GOTWILL:
#ifdef TELOPTS
	    tty_printf("[got WILL %s]\n", TELOPTSTR(buf[i]));
#endif
	    switch(buf[i]) {
	    case TELOPT_ECHO:
		/* host echoes, turn off echo here */
		/*
		 * turn off only for main session, since we do not want
		 * subsidiary session password entries to block anything
		 * in the main session
		 */
		if (is_main_sess())
		    linemode |= LM_NOECHO;
		sendopt(DO, buf[i]);
		break;
	    case TELOPT_SGA:
		/* this can't hurt */
		linemode |= LM_CHAR;
		set_spec_keys();
		sendopt(DO, buf[i]);
		break;
	    default:
		/* don't accept other options */
		sendopt(DONT, buf[i]);
		break;
	    }
	    state = NORMAL;
	    break;
	
	case GOTWONT:
#ifdef TELOPTS
	    tty_printf("[got WONT %s]\n", TELOPTSTR(buf[i]));
#endif
	    if(buf[i] == TELOPT_ECHO) {
		/* host no longer echoes, we do it instead */
		linemode &= ~LM_NOECHO;
	    }
	    /* accept any WONT */
	    sendopt(DONT, buf[i]);
	    state = NORMAL;
	    break;
	
	case GOTDO:
#ifdef TELOPTS
	    tty_printf("[got DO %s]\n", TELOPTSTR(buf[i]));
#endif
	    switch(buf[i]) {
	    case TELOPT_SGA:
		linemode |= LM_CHAR;
		set_spec_keys();
		/* FALLTHROUGH */
	    case TELOPT_TTYPE:
		sendopt(WILL, buf[i]);
		break;
	    case TELOPT_NAWS:
		sendopt(WILL, buf[i]);
		sending_size = 1;
		send_term_size();
		break;
	    default:
		/* accept nothing else */
		sendopt(WONT, buf[i]);
		break;
	    }
	    state = NORMAL;
	    break;
	
	case GOTDONT:
#ifdef TELOPTS
	    tty_printf("[got DONT %s]\n", TELOPTSTR(buf[i]));
#endif
	    if(buf[i] == TELOPT_SGA) {
		linemode &= ~LM_CHAR;
		set_spec_keys();
	    }
	    sendopt(WONT, buf[i]);
	    state = NORMAL;
	    break;
	
	case GOTSB:
	    if(buf[i] == IAC) {
		state = GOTSBIAC;
	    } else {
		if(subchars < MAXSUBOPT)
		    subopt[subchars++] = buf[i];
	    }
	    break;
	
	case GOTSBIAC:
	    if(buf[i] == IAC) {
		if(subchars < MAXSUBOPT)
		    subopt[subchars++] = IAC;
		state = GOTSB;
	    } else if(buf[i] == SE) {
		subopt[subchars] = '\0';
		dosubopt((char*)subopt);
		state = NORMAL;
	    } else {
		/* problem! I haven't the foggiest idea of what to do here.
		 * I'll just ignore it and hope it goes away. */
		tty_printf("Telnet problem: got IAC <%d> instead of IAC SE!\n",
		       buf[i]);
		state = NORMAL;
	    }
	    break;
	default:
	    syserr("Canthappen!");
	    break;
	}
    }
    return (char*)p - buffer;
}

/*
 * send an option negotiation
 * 'what' is one of WILL, WONT, DO, DONT
 */
static void sendopt(what, opt)
int what; unsigned opt;
{
    unsigned char buf[3];
    buf[0] = IAC; buf[1] = what; buf[2] = opt;
	if(send(sockfd, buf, 3, 0) == SOCKET_ERROR)
		syserr("write option");
#ifdef TELOPTS
    tty_printf("[sent %s %s]\n", (what == WILL) ? "WILL" :
	   (what == WONT) ? "WONT" :
	   (what == DO) ? "DO" : (what == DONT) ? "DONT" : "error",
	   TELOPTSTR(opt));
#endif
}

/*
 * Send current terminal size (RFC 1073)
 */
void send_term_size()
{
    static int base_fd=0;
    if (!base_fd)
        base_fd = sockfd;  /* Remember socket number of orig connection */

    if(sending_size) {
	static unsigned char buf[9] =
	    { IAC, SB, TELOPT_NAWS, 0, 0, 0, 0, IAC, SE };

	buf[3] = cols >> 8;
	buf[4] = cols & 0xff;
	buf[5] = lines >> 8;
	buf[6] = lines & 0xff;
	if(send(base_fd, buf, 9, 0) == SOCKET_ERROR)
		syserr("write term size");
#ifdef TELTOPS
	tty_printf("[term size %d %d]\n", cols, lines);
#endif
    }
}

/*
 * process suboptions.
 * so far, only terminal type is processed but future extensions are
 * window size, X display location, etc.
 */
static void dosubopt(str)
char *str;
{
    char buf[256], *term;
  
    if(str[0] == TELOPT_TTYPE) {
	if(str[1] == 1) {
	    /* 1 == SEND */
#ifdef TELOPTS
	    tty_printf("[got SB TERMINAL TYPE SEND]\n");
#endif
	    if(!(term = getenv("TERM"))) term = "unknown";
	    sprintf(buf, "%c%c%c%c%s%c%c", IAC, SB, TELOPT_TTYPE, 0,
		    term, IAC, SE);	/* 0 == IS */
		if(send(sockfd, buf, strlen(term) + 6, 0) == SOCKET_ERROR)
		syserr("write subopt to socket");
#ifdef TELOPTS
	    tty_printf("[sent SB TERMINAL TYPE IS %s]\n", term);
#endif
	}
    }
}


static char output_buffer[OUTBUFSIZE];
static int output_len = 0;	/* number of characters in output_buffer */
static int output_socket = -1;	/* to which socket buffer should be sent*/

/*
 * put data in the output buffer for transmission to the remote host
 */
void send_to_host(data)
char *data;
{
    int len, space;

//	tty_printf("send_to_host called for [%s]\n", data);

    sent += len = strlen(data);
    internal = -1;

    if (!(linemode & LM_NOECHO) && capfile)
        fprintf(capfile, "%s\n", data);

    if(sockfd != output_socket) { /* is there data to another socket? */
		if(output_socket >= 0)
			flush_output_buffer(); /* then flush it */
		output_socket = sockfd;
	}

    space = OUTBUFSIZE - output_len;
    
	if(space <= len) {  		/* not enough room? */
		bcopy(data, output_buffer + output_len, space);
		output_len = OUTBUFSIZE;
		flush_output_buffer();
		send_to_host(data + space);
    } else {
        data[len] = '\n';
		bcopy(data, output_buffer + output_len, len + 1);
		output_len += len + 1;
        data[len] = '\0';
		flush_output_buffer();
    }

}

/*
 * send all buffered data to the remote host
 */
void flush_output_buffer()
{
    int n;
    char *p = output_buffer;

//	tty_printf("flush outbuffer called, output_len %d, output_socket %d\n",
//			output_len, output_socket);

    if (output_len && output_socket == -1) {
      clearinpline(1);
      PRINT_INT ("#No open sessions. Use '#connect main <address> <port>' to start a session.\n");
      output_len = 0;
      showprompt();
      redraw_input();
      return;
    }
      
    while(output_len) {
//		tty_printf("Preparing to send %d bytes to host.\n", output_len);
		if((n = send(output_socket, p, output_len, 0)) == SOCKET_ERROR)
		syserr("write to socket");
	p += n;
	output_len -= n;
    }
}
