/*
 * term.c
 * terminal handling routines for cancan
 */

#ifdef WIN32

#ifndef VT100
#define VT100
#endif
#undef USETERMIO

#endif /* WIN32 */

#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <signal.h>
#include <time.h>

#ifdef USETERMIO
#ifdef APOLLO
#include "/sys5.3/usr/include/sys/termio.h"
#else
/*
 * including both termio.h and termios.h might be an overkill, and gives
 * many warnings, but seems to be necessary at times. works anyway.
 */
#include <termio.h>
#include <termios.h>
#endif
#else /* not USETERMIO */
#ifndef WIN32
#include <sys/ioctl.h>
#endif
#endif

#ifdef BSD_386
#include <sys/ioctl_compat.h>
#define O_ECHO ECHO
#define O_CBREAK CBREAK
#endif

/* int ioctl(); */

#ifdef __hpux  /* cc for HP-UX  SHOULD define this... */
#define TCGETS TCGETATTR
#define TCSETS TCSETATTR
#endif

#include "cancan.h"
#include "edit.h"
#include "term.h"

#ifndef VT100
/* termcap function declarations */
extern int tgetent();
extern int tgetnum();
extern int tgetflag();
extern char *tgetstr();
extern char *tgoto();
#endif

/* terminal escape sequences */
char kpadstart[CAPLEN], kpadend[CAPLEN], begoln[CAPLEN], clreoln[CAPLEN],
     leftcur[CAPLEN], rightcur[CAPLEN], curgoto[CAPLEN], inschar[CAPLEN],
     delchar[CAPLEN], clreoscr[CAPLEN], upcur[CAPLEN];
/* attribute changers: */
char modenorm[CAPLEN], backupmodenorm[CAPLEN], modebold[CAPLEN],
     modeinv[CAPLEN], modeuline[CAPLEN],
     modestandon[CAPLEN], modestandoff[CAPLEN];

int wrapglitch;			/* strange wrap behaviour (a la vt100) */
static int gotocost;		/* length of a typical tgoto-string */
int metakey;			/* terminal has meta key setting high bit */

#ifdef USETERMIO
#ifdef TCSETS
struct termios ttybsave;
#else
struct termio ttybsave;
#endif
#else
#ifndef WIN32
struct sgttyb ttybsave;
struct tchars tcsave;
struct ltchars ltcsave;
#endif
#endif

static int extract();

extern HANDLE outcons;

/*
 * Terminal handling routines:
 * These are one big mess of left-justified chicken scratches.
 * It should be handled more cleanly...but unix portability is what it is.
 */

/*
 * Set the terminal to character-at-a-time-without-echo mode, and save the
 * original state in ttybsave
 */
void set_terminal()
{
#ifndef WIN32
#ifdef USETERMIO
#ifdef TCSETS
    struct termios ttyb;
    ioctl(0, TCGETS, &ttyb);
#else
    struct termio ttyb;
    ioctl(0, TCGETA, &ttyb);
#endif
    ttybsave = ttyb;
    ttyb.c_lflag &= ~(ECHO|ICANON);
    ttyb.c_cc[VTIME] = 0;
    ttyb.c_cc[VMIN] = 1;
    /* disable the special handling of the suspend key (handle it ourselves) */
#ifdef VSUSP
    ttyb.c_cc[VSUSP] = 0;
#else
#ifdef SWTCH
    ttyb.c_cc[SWTCH] = 0;
#else
    ttyb.c_cc[SUSP] = 0;
#endif
#endif
#ifdef TCSETS
    ioctl(0, TCSETS, &ttyb);
#else
    ioctl(0, TCSETA, &ttyb);
#endif
#else /* not USETERMIO */
    struct sgttyb ttyb;
    struct ltchars ltc;
    ioctl(0, TIOCGETP, &ttybsave);
    ioctl(0, TIOCGETC, &tcsave);
    ioctl(0, TIOCGLTC, &ltcsave);
    ttyb = ttybsave;
    ttyb.sg_flags = (ttyb.sg_flags|O_CBREAK) & ~O_ECHO;
    ioctl(0, TIOCSETP, &ttyb);
    ltc = ltcsave;
    ltc.t_suspc = -1;
    ioctl(0, TIOCSLTC, &ltc);
#endif 
    tty_printf("%s", kpadstart); tty_flush();
#endif
}

/*
 * Reset the terminal to its original state
 */
void reset_terminal()
{
#ifndef WIN32
#ifdef USETERMIO
#ifdef TCSETS
    ioctl(0, TCSETS, &ttybsave);
#else
    ioctl(0, TCSETA, &ttybsave);
#endif
#else
    ioctl(0, TIOCSETP, &ttybsave);
    ioctl(0, TIOCSLTC, &ltcsave);
#endif
    tty_printf("%s", kpadend); tty_flush();
#endif
}

/*
 * enable/disable special keys depending on the current linemode
 */
void set_spec_keys()
{
#ifndef WIN32
#ifdef USETERMIO
    int i;
#ifdef TCSETS
    struct termios ttyb;
    ioctl(0, TCGETS, &ttyb);
#else
    struct termio ttyb;
    ioctl(0, TCGETA, &ttyb);
#endif
    if(linemode & LM_CHAR) 
    {
	/* char-by-char mode: disable all special keys and set raw mode */
	for(i = 0; i < NCCS; i++)
	    ttyb.c_cc[i] = 0;
	ttyb.c_oflag &= ~OPOST;
    }
    else 
    {
	/* line at a time mode: enable them, except suspend */
	for(i = 0; i < NCCS; i++)
	    ttyb.c_cc[i] = ttybsave.c_cc[i];
	/* disable the suspend key (handle it ourselves) */
#ifdef VSUSP
	ttyb.c_cc[VSUSP] = 0;
#else
#ifdef SWTCH
	ttyb.c_cc[SWTCH] = 0;
#else
	ttyb.c_cc[SUSP] = 0;
#endif
#endif
	/* set cooked mode */
	ttyb.c_oflag |= OPOST;
    }
#ifdef TCSETS
    ioctl(0, TCSETS, &ttyb);
#else
    ioctl(0, TCSETA, &ttyb);
#endif
#else /* not USETERMIO */
    struct tchars tc = {-1, -1, -1, -1, -1, -1};
    struct ltchars ltc = {-1, -1, -1, -1, -1, -1};
    struct sgttyb ttyb;
    ioctl(0, TIOCGETP, &ttyb);
    if(linemode & LM_CHAR) 
    {
	/* char-by-char mode: set RAW mode*/
	ttyb.sg_flags |= RAW;
    }
    else 
    {
	/* line-at-a-time mode: enable spec keys, disable RAW */
	tc = tcsave;
	ltc = ltcsave;
	ltc.t_suspc = -1;	/* suspend key remains disabled */
	ttyb.sg_flags &= ~RAW;
    }
    ioctl(0, TIOCSETP, &ttyb);
    ioctl(0, TIOCSETC, &tc);
    ioctl(0, TIOCSLTC, &ltc);
#endif
#endif
}

/*
 * read termcap definitions
 */
void settermcap()
{
#if !defined(VT100) || defined(TELNETBUG)
    char *term = getenv("TERM");
    if(!term) {
	tty_printf("$TERM not set\n"); exitcancan();
    }
#endif
    /* hard-coded vt100 features if no termcap: */
    strcpy(begoln, "\r");
    strcpy(clreoln, "\033[K");
    strcpy(clreoscr, "\033[J");
    strcpy(leftcur, "\033[D");
    strcpy(rightcur, "\033[C");
    strcpy(upcur, "\033[A");
    strcpy(modenorm, "\033[m");
    strcpy(modebold, "\033[1m");
    strcpy(modeinv, "\033[7m");
    strcpy(modeuline, "\033[4m");
    /* not all of the above attributes exist on all vt100s */
    cols = 80;
#ifdef LINES
    lines = LINES;
#else
    lines = 25;
#endif
    *kpadstart = *kpadend = 0;
    //    if (!*lookup_key("left")) add_keynode("left", "\033[D", prev_char, NULL);
    //    if (!*lookup_key("right")) add_keynode("right", "\033[C", next_char, NULL);
    //    if (!*lookup_key("up")) add_keynode("up", "\033[A", prev_line, NULL);
    //    if (!*lookup_key("down")) add_keynode("down", "\033[B", next_line, NULL);
    wrapglitch = 1;
    metakey = 0;		/* vt100 has no meta key */
    gotocost = 7;		/* average for vt100 cursor movement? */
    
    strcpy(backupmodenorm, modenorm);
#ifdef TELNETBUG
    /* might be NCSA Telnet 2.2 for PC, which doesn't reset colours */
    if(strncmp(term, "vt10", 4) == 0) {
        stty_printf(modenorm, "\033[;%c%d;%s%dm", 
		    DEFAULTFG<LOWCOLOURS ? '3' : '9',  DEFAULTFG % LOWCOLOURS,
		    DEFAULTBG<LOWCOLOURS ? "4" : "10", DEFAULTBG % LOWCOLOURS);
    }
#endif
}

/*
 * add the default keypad bindings to the list
 */
void default_keybindings()
{
    char buf[] = "\033x";
    /*
     * Note: termcap doesn't have sequences for the numeric keypad, so we just
     * assume they are the same as for a vt100.
     */
#ifdef WIN32
    add_keynode("kp2", (buf[1] = VK_NUMPAD2, buf), key_run_command, "s");
    add_keynode("kp3", (buf[1] = VK_NUMPAD3, buf), key_run_command, "d");
    add_keynode("kp4", (buf[1] = VK_NUMPAD4, buf), key_run_command, "w");
    add_keynode("kp5", (buf[1] = VK_NUMPAD5, buf), key_run_command, "exits");
    add_keynode("kp6", (buf[1] = VK_NUMPAD6, buf), key_run_command, "e");
    add_keynode("kp8", (buf[1] = VK_NUMPAD8, buf), key_run_command, "n");
    add_keynode("kp9", (buf[1] = VK_NUMPAD9, buf), key_run_command, "u");
    add_keynode("kp7", (buf[1] = VK_NUMPAD7, buf), key_run_command, "look");
#endif
}

void scrollup();
void scrolldown();

/*
 * initialize the key binding list
 */
void init_keybindings()
{
    char buf2[] = "\033x";
    char buf[2];
    buf[1] = '\0';
    /* standard emacs-like key bindings */
    /* (should perhaps be sorted in decreasing frequency of use) */
    add_keynode("return", "\r", enter_line, NULL);
    add_keynode("lf", "\n", enter_line, NULL);
    add_keynode("tab", "\t", complete_word, NULL);
    add_keynode("bs", "\b", del_char_left, NULL);
    add_keynode("backsp", "\177", del_char_left, NULL);
    add_keynode("delete", (buf2[1] = VK_DELETE, buf2), del_char_right, NULL);
    //    add_keynode("M-tab", "\033\t", complete_line, NULL);
    add_keynode("left", (buf2[1] = VK_LEFT, buf2) , prev_char, NULL);
    add_keynode("right", (buf2[1] = VK_RIGHT, buf2) , next_char, NULL);
    add_keynode("up", (buf2[1] = VK_UP, buf2) , prev_line, NULL);
    add_keynode("down", (buf2[1] = VK_DOWN, buf2) , next_line, NULL);
    add_keynode("pageup", (buf2[1] = VK_PRIOR, buf2) , scrollup, NULL);
    add_keynode("pagedown", (buf2[1] = VK_NEXT, buf2) , scrolldown, NULL);
    add_keynode("^W", (*buf = CONTROL('w'), buf) , enter_into_history,	NULL);
    add_keynode("^Q", (*buf = CONTROL('q'), buf) , clear_line, NULL);
    add_keynode("^D", (*buf = CONTROL('d'), buf) , del_char_right, NULL);
    add_keynode("^A", (*buf = CONTROL('a'), buf) , beginning_of_line, NULL);
    add_keynode("^E", (*buf = CONTROL('e'), buf) , end_of_line, NULL);
    add_keynode("^K", (*buf = CONTROL('k'), buf) , kill_to_eol, NULL);
    add_keynode("^T", (*buf = CONTROL('t'), buf) , transpose_chars, NULL);
    add_keynode("^L", (*buf = CONTROL('l'), buf) , redraw_input_line, NULL);
    add_keynode("^Z", (*buf = CONTROL('z'), buf) , suspend_cancan, NULL);
    //    add_keynode("M-B", "\033b", prev_word, NULL);
    //    add_keynode("M-F", "\033f", next_word, NULL);
    //    add_keynode("M-bs", "\033\b", del_word_left, NULL);
    //    add_keynode("M-backsp", "\033\177", del_word_left, NULL);
    //    add_keynode("M-D", "\033d", del_word_right, NULL);
}

#ifndef VT100
/*
 * extract the termcap 'cap' and put it in buf. Exit if critic != 0 and
 * 'cap' doesn't exist. Return 1 if 'cap' found, 0 if not.
 */
static int extract(cap, buf, critic)
char *cap, *buf;
int critic;
{
    char *p, b[80];
    
    p = b; *b = 0;
    if(!tgetstr(cap, &p)) {
	if(critic) {
	    fprintf(stderr,
		    "Your terminal %s is not powerful enough, missing '%s'.\n",
		    getenv("TERM"), cap);
	    exit(1);
	}
	*p = '\0';
	return 0;
    }
    /*
     * Remove the padding information. We assume that no terminals need
     * padding nowadays. At least it makes things much easier.
     */
    strcpy(buf, b + strspn(b, "0123456789*"));
    *p = '\0';
    return 1;
}
#endif /* VT100 */

/*
 * position the cursor using absolute coordinates
 * note: does not flush the output buffer
 */
#ifndef WIN32
void tty_gotoxy(col, line)
int col, line;
{
    
    tty_printf("\033[%d;%dH", line + 1, col + 1);
}
#endif

/*
 * optimized cursor movement (no output flush)
 * from (fromline, fromcol) to (toline, tocol)
 */
void optmove(fromline, fromcol, toline, tocol)
int fromline, fromcol, toline, tocol;
{
#ifndef WIN32
    static char buf[LINELEN];
    int cost = 0, i, dist;
    /*
     * First, move vertically to the correct line, then horizontally
     * to the right column. If this turns out to be fewer characters
     * than a direct cursor positioning (gotoxy), use that.
     */
    
    if(fromline > toline) {
	if(*upcur) {
	    for(i = 0; i < (fromline - toline); i++) {
		strcpy(buf + cost, upcur);
		cost += strlen(upcur);
	    }
	} else {
	    /* we can't move cursor locally up - give up optimization */
	    tty_gotoxy(tocol, toline);
	    return;
	}
    } else {
	cost = toline - fromline;
	for(i = 0; i < cost; i++) buf[i] = '\n';
	if(cost > 0) fromcol = 0; /* LF is mapped to CRLF on output */
    }
    if(fromcol > tocol) {
	dist = fromcol - tocol;
	if(strlen(leftcur) * dist <= strlen(begoln) + tocol) {
	    for(i = 0; i < dist; i++) {
		strcpy(buf + cost, leftcur);
		cost += strlen(leftcur);
	    }
	} else {
	    strcpy(buf + cost, begoln);
	    cost += strlen(begoln);
	    fromcol = 0;
	}
    }
    if(fromcol < tocol) {
	if(*edattrbeg) {
	    /* hiliting in effect: easy way out, just use gotoxy */
	    tty_gotoxy(tocol, toline);
	    return;
	}
	if(toline == line0 && fromcol < col0) {
	    /* assume that if fromcol < col0, then tocol >= col0 */
	    strcpy(buf + cost, promptstr + fromcol);
	    cost += col0 - fromcol;
	    fromcol = col0;
	}
	strncpy(buf + cost, edbuf + (toline - line0) * (cols - 1) - col0
		+ fromcol, tocol - fromcol);
	cost += tocol - fromcol;
    }
    if(cost < gotocost) {
	buf[cost] = '\0';
	tty_puts(buf);
    } else
#endif
	tty_gotoxy(tocol, toline);
    
}

/*
 * clear current input line (prompt ==1 if to clear the prompt)
 * cursor is left right after the prompt
 */
void clearinpline(prompt)
int prompt;
{
    int newcol = prompt ? 0 : col0;
    if (edlen || prompt) {
	optmove(CURLINE(pos), CURCOL(pos), line0, newcol);
	if (line0 == lines - 1)
	    tty_puts(clreoln);
	else
	    tty_puts(clreoscr);
	tty_flush();
    }
    col0 = newcol;
}

/*
 * Redraw the input line and put the cursor at the current position.
 * The cursor is assumed to be directly after the prompt.
 */
void redraw_input()
{
    int i, oldline0;
    if(edlen) {
	oldline0 = line0;
	if(edlen < cols - col0 - 1) {
	    tty_printf("%s%s", edattrbeg, edbuf);
	} else {
	    tty_printf("%.*s", cols - col0 - 1, edbuf);
	    for(i = cols - col0 - 1; i <= edlen; i += cols - 1)
		tty_printf("\n%.*s", cols - 1, edbuf + i);
	}
	line0 = lines - (edlen + col0) / (cols - 1) - 1;
	if(oldline0 < line0) line0 = oldline0;
	optmove(CURLINE(edlen), CURCOL(edlen), CURLINE(pos), CURCOL(pos));
    }
    tty_flush();
}

/*
 * delete n characters at current position (the position is unchanged)
 */
void deletechars(n)
int n;
{
    int i, j, cl, cc, endline;
    
    for(i = 0; i < n; i++) strdel(edbuf + pos);
    edlen -= n;
    
    /*
     * Write out the necessary changes. This code is a hell lot more
     * complicated than it need to be, and I am still not sure it works well
     * in all cases.
     */
    cc = CURCOL(pos); cl = CURLINE(pos);
    i = pos; endline = CURLINE(edlen);
    for(;;) 
    {
	/* no delete character, alas */
	if(cl < endline) 
	{
	    tty_printf("%.*s", cols - 1 - cc, edbuf + i);
	    cc = cols - 1;
	}
	else 
	{
	    tty_printf("%s%s", edbuf + i, clreoln);
	    cc = CURCOL(edlen);
	    break;
	}
	cl++; cc = 0;
	i = (cl - line0) * (cols - 1) - col0;
	tty_putc('\n');
    }
    optmove(cl, cc, CURLINE(pos), CURCOL(pos));
    tty_flush();
}

/*
 * insert n characters at current position. The position is set to after
 * the inserted characters.
 */
void insertchars(str, n)
char *str; int n;
{
    int i, j, k, cl, cc, endline;
    char *prtbuf, starbuf[1024];
    for(i = 0; i < n; i++) strins(edbuf + pos + i, str[i]);
    edlen += n;
    
    if(linemode & LM_NOECHO) {
	for(i = 0; i < strlen(edbuf); i++)
	    starbuf[i] = (*(edbuf+i) < ' ' ||
			  *(edbuf+i) > '~') ? *(edbuf+i) : '*';
	starbuf[i] = 0;
	prtbuf = starbuf;
    } else
	prtbuf = edbuf;
    
    cc = CURCOL(pos); cl = CURLINE(pos); endline = CURLINE(edlen);
    i = pos;
    for(;;) 
    {
	if(*inschar) 
	{
	    /*
	     * terminal can insert char, use it.
	     * The 3 is just ad-hockery: it is hard to calculate when inschar
	     * costs more than simply outputting the rest of the line, but
	     * it's not that important really.
	     */
	    if(cl == endline && edlen - i < 3 * n) 
	    {
		tty_puts(prtbuf + i);
		cc = CURCOL(edlen);
	    }
	    else 
	    {
		k = (cols - 1 - cc < n) ? cols - 1 - cc : n;
		for(j = 0; j < k; j++) tty_puts(inschar);
		tty_printf("%.*s", k, prtbuf + i);
		cc += k;
	    }
	    if(cl < endline) 
	    {
		/*
		 * we inserted things into last column that isn't used, so
		 * we'd better delete that one to make everything look pretty.
		 */
		optmove(cl, cc, cl, cols - 1);
		tty_puts(clreoln);
	    }
	    else
		break;
	}
	else 
	{
	    /* no inschar, print the whole line again */
	    if(cl < endline) 
	    {
		tty_printf("%.*s", cols - 1 - cc, prtbuf + i);
		cc = cols - 1;
	    }
	    else 
	    {
		tty_puts(prtbuf + i);
		cc = CURCOL(edlen);
		break;
	    }
	}
	cl++; cc = 0;
	if(cl == lines) 
	{
	    cl--;
	    line0--;
	    endline--;
	}
	i = (cols - 1) * (cl - line0) - col0;
	tty_putc('\n');
    }
    pos += n;
    optmove(cl, cc, CURLINE(pos), CURCOL(pos));
    tty_flush();
    
}

/*
 * window size change signal handler:
 * we assume that the bottom line is always the bottom line
 */
#ifndef WIN32
void winchsig()
{
    struct winsize wsiz;
    
    /* if ioctl fails or gives silly values, don't change anything */
    if(ioctl(0, TIOCGWINSZ, &wsiz) == 0 && wsiz.ws_row && wsiz.ws_col) {
	lines = wsiz.ws_row;
	cols = wsiz.ws_col;
	if (sockfd != -1)
	    send_term_size();
    }
    /* reinstall signal handler for system V unices */
    signal(SIGWINCH, winchsig);
}
#endif

#ifdef WIN32

/* Win32 specific stuff */

/*
 * a replacement for UNIX gettimeofday()
 * with at least millisecond accuracy.
 * We ignore the timezone.
 */
void gettimeofday(tv, dummy) 
vtime *tv;
void *dummy;
{
    SYSTEMTIME stim;
    GetLocalTime(&stim);
    tv->tv_sec = time(NULL);
    tv->tv_usec = stim.wMilliseconds * 1000;
}

extern FILE *logfile;

typedef enum {
    ES_norm, ES_esc, ES_square, ES_parms, ES_err 
} ES_state;

static int parms[10], nparms;

static char blank_buf[BUFSIZE];
static int buf_ready = 0;

static void cons_init_buf() {
    memset(blank_buf, ' ', BUFSIZE);
    buf_ready = 1;
}


static void cons_gotoxy_rel(x, y)
int x, y;
{
    CONSOLE_SCREEN_BUFFER_INFO myconsinfo;
    GetConsoleScreenBufferInfo(outcons, &myconsinfo);    
    
    myconsinfo.dwCursorPosition.X += x;
    myconsinfo.dwCursorPosition.Y += y;
    SetConsoleCursorPosition(outcons, myconsinfo.dwCursorPosition);
}

void tty_gotoxy(x, y)
int x, y;
{ 
    COORD cpos;
    cpos.X = x;
    cpos.Y = y;
    SetConsoleCursorPosition(outcons, cpos);
}

/* clear to end of screen */
static void cons_escJ() {
    int len, y;
    WORD attr;
    COORD cpos;
    CONSOLE_SCREEN_BUFFER_INFO myconsinfo;
    
    if (!buf_ready) {
	cons_init_buf();
    }    

    GetConsoleScreenBufferInfo(outcons, &myconsinfo);    
    
    attr = myconsinfo.wAttributes;
    len = cols * (lines - myconsinfo.dwCursorPosition.Y)
          - myconsinfo.dwCursorPosition.X - 1;

    SetConsoleTextAttribute(outcons, FOREGROUND_RED | FOREGROUND_GREEN | FOREGROUND_BLUE);
    WriteConsole(outcons, blank_buf, len, &y, 0);
    SetConsoleCursorPosition(outcons, myconsinfo.dwCursorPosition);
    SetConsoleTextAttribute(outcons, attr);
}

/* clear to end of line */
static void cons_escK() {
    int len, y;
    CONSOLE_SCREEN_BUFFER_INFO myconsinfo;
    
    if (!buf_ready) {
	cons_init_buf();
    }

    GetConsoleScreenBufferInfo(outcons, &myconsinfo);    
    
    len = cols - myconsinfo.dwCursorPosition.X - 1;

    SetConsoleTextAttribute(outcons, FOREGROUND_RED | FOREGROUND_GREEN | FOREGROUND_BLUE);
    WriteConsole(outcons, blank_buf, len, &y, 0);
    SetConsoleCursorPosition(outcons, myconsinfo.dwCursorPosition);
    SetConsoleTextAttribute(outcons, myconsinfo.wAttributes);
}

static void cons_escm() {
    static int inverse = 0;
    int fgcol, bgcol, bold, blink, i, p, col;
    CONSOLE_SCREEN_BUFFER_INFO myconsinfo;
    
    GetConsoleScreenBufferInfo(outcons, &myconsinfo);    
    fgcol = myconsinfo.wAttributes & 0x0f;
    bold  = fgcol & FOREGROUND_INTENSITY ? 1 : 0;
    bgcol = myconsinfo.wAttributes & 0xf0;
    blink = bgcol & BACKGROUND_INTENSITY ? 1 : 0;

    for (i=0; i<nparms; i++) {
	switch (p = parms[i]) {
	  case 0:
	    fgcol = FOREGROUND_RED | FOREGROUND_GREEN | FOREGROUND_BLUE;
	    bgcol = bold = blink = inverse = 0;
	    break;
	  case 1:
	  case 4:
	    bold = 1;
	    break;
	  case 5:
	    blink = 1;
	    break;
	  case 7:
	    inverse = 1;
	    break;
	  case 21:
	  case 24:
	    bold = 0;
	    break;
	  case 25:
	    blink = 0;
	    break;
	  default:
	    if ((p >= 30 && p <= 37) ^ inverse) {
		p %= 10;
		fgcol = (p&1?FOREGROUND_RED:0) | (p&2?FOREGROUND_GREEN:0) |
		  (p&4?FOREGROUND_BLUE:0);
	    } else if ((p >= 40 && p <= 47) ^ inverse) {
		p %= 10;
		bgcol = (p&1?BACKGROUND_RED:0) | (p&2?BACKGROUND_GREEN:0) |
		  (p&4?BACKGROUND_BLUE:0);
	    }
	    break;
	}
    }
    
    if (!inverse) {
      fgcol |= (bold?FOREGROUND_INTENSITY:0);
      bgcol |= (blink?BACKGROUND_INTENSITY:0);
    } else {
      fgcol |= (blink?FOREGROUND_INTENSITY:0);
      bgcol |= (bold?BACKGROUND_INTENSITY:0);    
    }
      
    SetConsoleTextAttribute(outcons, myconsinfo.wAttributes & 0xff00 |
			    (bgcol | fgcol));
}

static void tty_newline()
{
    int y;
    CONSOLE_SCREEN_BUFFER_INFO myconsinfo;
    
    GetConsoleScreenBufferInfo(outcons, &myconsinfo);
    SetConsoleTextAttribute(outcons, FOREGROUND_RED | FOREGROUND_GREEN | FOREGROUND_BLUE);
    WriteConsole(outcons, "\n", 1, &y, 0);
    SetConsoleTextAttribute(outcons, myconsinfo.wAttributes);
}

void tty_putc(ic)
int ic;
{
    int y;
    char c = ic;
    if (c == '\n')
      tty_newline();
    else
      WriteConsole(outcons, &c, 1, &y, 0);
}

void tty_puts(s)
char *s;
{
    char *start, c;
    int len, n, y;
    static ES_state state = ES_norm;

    /*
     * if there is an escape sequence decipher it,
     * transforming ANSI stuff into Win32 Console calls
     */
    while((c = *s)) {
	switch (state) {
	 case ES_norm:
	    if (c >= ' ') {
		start = s; len = 0;
		while (*s && *s >= ' ')
		  s++, len++;
		WriteConsole(outcons, start, len, &y, 0);
	    }
	    
	    if (!(c = *s))
	      break;

	    /* c < ' ' here */
	    if (c != '\033') {
		if (c == '\n')
		  tty_newline();
		else
		  WriteConsole(outcons, &c, 1, &y, 0);
		++s;
		break;
	    }
	    /* c == '\033' here */
	    state = ES_esc;
	    c = *++s;
	    /* fallthrough */
	 case ES_esc:
	    /* we just got the ESC, SHOULD be a left-bracket now */
	    if (c != '[') {
		state = ES_norm;
		break;
	    }
	    state = ES_square;
	    c = *++s;
	    /* fallthrough */
	 case ES_square:
	    /* we have a left bracket, look for parameters */
	    nparms = 0;
	    while (c && (isdigit(c) || c == ';')) {
		if (isdigit(c)) {
		    if (nparms < 10) {
			parms[nparms] = atoi(s);
			nparms++;
		    }
		    while (isdigit(c))
		      c = *++s;
		    if (c == ';')
		      c = *++s;
		    
		} else if (c == ';') {
		    if (nparms < 10) {
			parms[nparms] = 0;
			nparms++;
		    }
		    c = *++s;
		}
	    }
	    if (nparms == 0)
	      parms[nparms++] = 0;
	    
	    state = ES_parms;
	    /* fallthrough */
	 case ES_parms:
	    switch (c) {
	     case 'A': /* cursor up */
		cons_gotoxy_rel(0,-1);
		++s;
		break;
	     case 'B': /* cursor down */
		cons_gotoxy_rel(0,+1);
		++s;
		break;
	     case 'C': /* cursor right */
		cons_gotoxy_rel(+1,0);
		++s;
		break;
	     case 'D': /* cursor left */
		cons_gotoxy_rel(-1,0);
		++s;
		break;
	     case 'H': /* cursor gotoxy */
		tty_gotoxy(parms[0], parms[1]);
		++s;
		break;
	     case 'J': /* clear to end of screen */
		cons_escJ();
		++s;
		break;
	     case 'K': /* clear to end of line */
		cons_escK();
		++s;
		break;
	     case 'm': /* colors and attributes */
		cons_escm();
		++s;
		break;
	     default:
		break;
	    }
	    state = ES_norm;
	}
    }
}    

/*
 * MUST use ANSI prototype here...
 */
void tty_printf(char *format, ...)
{
    char buf[BUFSIZE*100];
    va_list ap;

    va_start(ap, format);
    vsprintf(buf, format, ap);
    tty_puts(buf);
}

#endif
