/*
 * Xeval.c
 * functions for inline calculator
 *
 * (created: Massimiliano Ghilardi (Cosmos), Jan 15th, 1995)
 */

#include <stdio.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#ifndef WIN32
#include <sys/time.h>
#else
#define NORANDOM 1
#endif
#include <limits.h>

#include "cancan.h"
#include "cmd.h"
#include "xeval.h"

#ifndef RAND
#define random lrand48
#define srandom srand48
#endif

char *first_valid();

char *error_msg[] = {
  "unknown error",
  "math stack overflow",
  "math stack underflow",
  "expression syntax",
  "operator expected",
  "value expected",
  "division by zero",
  "operand or index out of range",
  "missing right parenthesis",
  "missing left parenthesis",
  "not enough free memory",
  "operator is not (yet) implemented",
  "infinite loop",
  "numeric value expected",
  "string expected",
  "stack overflow",
  "stack underflow",
  "missing label",
  "missing separator ';'",
  "attempted to execute recursive #history",
  "user break",
  "too many defined variables",
  "undefined variable",
};

operator_list op_list[] = {
  { 0, 0,     0,	  ""     },     /* null operator */

  { 1, LEFT,  BINARY,     ","    },

  { 2, RIGHT, BINARY,     "="    },
  { 2, RIGHT, BINARY,     "||="  },
  { 2, RIGHT, BINARY,     "^^="  },
  { 2, RIGHT, BINARY,     "&&="  },
  { 2, RIGHT, BINARY,     "|="   },
  { 2, RIGHT, BINARY,     "^="   },
  { 2, RIGHT, BINARY,     "&="   },
  { 2, RIGHT, BINARY,     "<<="  },
  { 2, RIGHT, BINARY,     ">>="  },
  { 2, RIGHT, BINARY,     "+="   },
  { 2, RIGHT, BINARY,     "-="   },
  { 2, RIGHT, BINARY,     "*="   },
  { 2, RIGHT, BINARY,     "/="   },
  { 2, RIGHT, BINARY,     "%="   },

  { 3, LEFT,  BINARY,     "||"   },

  { 4, LEFT,  BINARY,     "^^"   },

  { 5, LEFT,  BINARY,     "&&"   },

  { 6, LEFT,  BINARY,     "|"    },

  { 7, LEFT,  BINARY,     "^"    },

  { 8, LEFT,  BINARY,     "&"    },

  { 9, LEFT,  BINARY,     "<"    },
  { 9, LEFT,  BINARY,     "<="   },
  { 9, LEFT,  BINARY,     ">"    },
  { 9, LEFT,  BINARY,     ">="   },
  { 9, LEFT,  BINARY,     "=="   },
  { 9, LEFT,  BINARY,     "!="   },

  {10, LEFT,  BINARY,     "<<"   },
  {10, LEFT,  BINARY,     ">>"   },

  {11, LEFT,  BINARY,     "+"    },
  {11, LEFT,  BINARY,     "-"    },

  {12, LEFT,  BINARY,     "*"    },
  {12, LEFT,  BINARY,     "/"    },
  {12, LEFT,  BINARY,     "%"    },

  {14, LEFT,  BINARY,     ":<"   },
  {14, LEFT,  BINARY,     ":>"   },
  {14, LEFT,  BINARY,     "<:"   },
  {14, LEFT,  BINARY,     ">:"   },
  {14, LEFT,  BINARY,     ".<"   },
  {14, LEFT,  BINARY,     ".>"   },
  {14, LEFT,  BINARY,     "<."   },
  {14, LEFT,  BINARY,     ">."   },
  {14, LEFT,  BINARY,     ":"    },
  {14, LEFT,  BINARY,     "."    },
  {14, LEFT,  BINARY,     "?"    },

  { 0, 0,     0,	  ""     },     /* null operator */

  { 0, RIGHT, PRE_UNARY,  "("    },
  { 0, RIGHT, POST_UNARY, ")"    },

  {13, RIGHT, PRE_UNARY,  "!"    },
  {13, RIGHT, PRE_UNARY,  "~"    },
  {13, RIGHT, PRE_UNARY,  "++"   },
  {13, RIGHT, POST_UNARY, "++"   },
  {13, RIGHT, PRE_UNARY,  "--"   },
  {13, RIGHT, POST_UNARY, "--"   },
  {13, RIGHT, PRE_UNARY,  "*"    },
  {13, RIGHT, PRE_UNARY,  "%"    },
  {13, RIGHT, PRE_UNARY,  "rand" },

  {14, LEFT,  PRE_UNARY,  ":?"   },
  {14, LEFT,  PRE_UNARY,  ".?"   },

  {15, RIGHT, PRE_UNARY,  "+"    },
  {15, RIGHT, PRE_UNARY,  "-"    },
  {15, RIGHT, PRE_UNARY,  "@"    },
  {15, RIGHT, PRE_UNARY,  "$"    },

  { 0, 0,     PRE_UNARY,  ""     },    /* null operator */
  { 0, 0,     POST_UNARY, ""     }     /* null operator */
};

stack stk;
char *line;
int depth;
int error;
int is_fromfile;

void my_memmove();
int my_strstr();
int my_strrstr();
int push_op();
int pop_op();
int push_obj();
int pop_obj();
int check_operator();
int check_object();
void check_delete();
int exe_op();
int whichfirst();
int compare_and_unload();
int eval();

void my_memmove(dst, src, len)
char *dst, *src; int len;
{
  if (len <= 0)
    return;

  if (dst < src)
    while(len)
      len--, *(dst)++ = *(src)++;
  else if (dst > src) {
    dst += len - 1;
    src += len - 1;
    while (len)
      len--, *(dst)-- = *(src)--;
  }
}

void print_error(err_num)
int err_num;
{
  tty_printf("#error: %s.\n", error_msg[err_num]);
}

/*
 * Scan s1 for the first occurrence of one of the characters in s2,
 * and return -1 if none of them is found.
 */

int my_strstr(s1, s2)
char *s1, *s2;
{
  int i=0;

  while(*s1 && !strchr(s2, *s1))
    i++, s1++;

  if (!*s1) i=-1;
  return i;
}

/*
 * Scan s1 for the last occurrence (i<=strlen(s1)-1, last valid char)
 * of one of the characters in s2,
 * and return -1 if none of them is found.
 */

int my_strrstr(s1, s2, i)
char *s1, *s2;
int i;
{
  char *s=s1+i;

  while(i>=0 && !strchr(s2, *s))
    i--, s--;

  return i;
}

int push_op(op)
operator *op;
{
  if (stk.curr_op<MAX_STACK) {
    stk.op[++stk.curr_op]=*op;
    return 1;
  }
  else {
    print_error(error=STACK_OV_ERROR);
    return 0;
  }
}

int pop_op(op)
operator *op;
{
  if (stk.curr_op>=0) {
    *op=stk.op[stk.curr_op--];
    return 1;
  }
  else {
    print_error(error=STACK_UND_ERROR);
    return 0;
  }
}

int push_obj(obj)
object *obj;
{
  object *tmp;

  int curr=stk.curr_obj;

  if (curr<MAX_STACK) {
    tmp = stk.obj + (stk.curr_obj = ++curr);
    memcpy(tmp, obj, sizeof(object));
    return 1;
  }
  else {
    print_error(error=STACK_OV_ERROR);
    return 0;
  }
}

int pop_obj(obj)
object *obj;
{
  object *tmp;

  int curr=stk.curr_obj;

  if (curr>=0) {
    tmp = stk.obj + curr;
    stk.curr_obj--;
    memcpy(obj, tmp, sizeof(object));
    return 1;
  }
  else {
    print_error(error=STACK_UND_ERROR);
    return 0;
  }
}

int check_operator(side, op, mindepth)
char side; operator *op; int mindepth;
{
  int i, max, match, len;
  char *name, c, d;

  if (!(c=*line) || c == CMDSEP) {
    *op = side==BINARY ?
	(operator)null
	: side==LEFT ?
	    (operator)pre_null
	    : (operator)post_null;

    return 1;
  }
  else if ((c=='$' || c=='@') && (d=line[1]) && (isalpha(d) || d=='_'))
    return 0;           /* Danger! found named variable */

  else if (side==LEFT && c=='(') {
    line++;
    depth++;
    *op=(operator)left_paren;
    return 1;
  }
  else if (side==RIGHT && c==')') {
    if (--depth >= mindepth) {
      line++;
      *op=(operator)right_paren;
    }
    else 		   /* exit without touching the parenthesis */
      *op=(operator)post_null;
    return 1;
  }

  for(max=match=0, i=(side==BINARY ? 1 : LOWEST_UNARY_CODE);
      *(name = op_list[i].name); i++)
    if ((len=strlen(name)) > max &&
	(side==BINARY || side==op_list[i].syntax) &&
	!strncmp(line, name, (size_t)len)) {
      match=i;
      max=len;
    }

  if (match) {
    *op=(operator)match;
    line+=max;
    return 1;
  }
  else {
    *op= side==BINARY ? null : side==PRE_UNARY ? pre_null : post_null;
    if (side==BINARY)
      print_error(error=NO_OPERATOR_ERROR);
  }
  return 0;
}

int check_object(obj)
object *obj;
{
  long i=0;
  char c, *end;

  if(isdigit(c=*line)) {
    while(isdigit(c)) {
      i*=10;
      i+=(c-'0');
      c=*++line;
    }
    obj->type=TYPE_NUM;
    obj->num=i;
    i=1;
  }
  else if(c=='\"') {
    end=first_valid(++line, '\"');
    if (*end) {
      if (obj->txt=(char *)malloc(PARAMLEN)) {
        obj->type=TYPE_TXT;
	*end=0;
        strncpy(obj->txt, line, PARAMLEN);
	obj->txt[PARAMLEN - 1] = '\0';
        unescape(obj->txt);
	*end='\"';
        i=1;
        line=end+1;
      }
      else
        print_error(error=NO_MEM_ERROR);
    }
  }
  else if (!strncmp(line, "timer", 5)) {
    obj->type = TYPE_NUM;
    obj->num = diff_vtime(&now, &ref_time);
    line += 5;
    i = 1;
  }
  else if (!strncmp(line, "map", 3)) {
    obj->type = TYPE_TXT;
    if (obj->txt = (char *)malloc(PARAMLEN)) {
      print_automap(obj->txt);
      line += 3;
      i = 1;
    } else
      print_error(error=NO_MEM_ERROR);
  }
  else if ((c=='$' || c=='@') && (c=line[1]) && (isalpha(c) || c=='_')) {
    varnode *named_var;                       /* Found named variable */

    if (*(line++) == '@') {
      i = 0;
      obj->type = TYPE_NUM_VAR;
    }
    else {
      i = 1;
      obj->type = TYPE_TXT_VAR;
    }
    end = line + 1;
    while ((c=*end) && (isalpha(c) || c=='_' || isdigit(c)))
	end++;
    c = *end; *end = '\0';
    if (!(named_var = *lookup_varnode(line, (char)i))) {
      named_var = add_varnode(line, (char)i);
      if (error)
	return 0;
      if (!is_fromfile && echo_int) {
	PRINT_INT ("#new variable: %s\n", line - 1);
      }
    }
    *end = c;
    line = end;
    obj->num = named_var->index;
    i = 1;
  }
  else
    print_error(error=NO_VALUE_ERROR);

  return (int)i;
}

void check_delete(obj)
object *obj;
{
  if (obj->type==TYPE_TXT && obj->txt)
    free(obj->txt);
}

int exe_op(op)
operator *op;
{
  object o1, o2, *p=NULL;
  long *l;
  char *src, *dst, *start;
  int ret=0, i=0, j=0, danger=0;

  switch (*op) {
    case (operator)comma:
      if (pop_obj(&o2) && pop_obj(&o1));
      else if (error) return 0;

      p=&o2;
      ret=1;
      break;
    case (operator)eq:
      if (pop_obj(&o2) && pop_obj(&o1));
      else if (error) return 0;

      if (o2.type==TYPE_NUM_VAR) {
        o2.num = *var[(int)o2.num+NUMVAR].num;
        o2.type = TYPE_NUM;
      }

      if (o1.type==TYPE_NUM_VAR && o2.type==TYPE_NUM) {
	*var[(int)o1.num+NUMVAR].num = o2.num;
        p=&o2;
        ret=1;
      }
      else if (o1.type==TYPE_TXT_VAR &&
		(o2.type==TYPE_TXT || o2.type==TYPE_TXT_VAR)) {

        if (o2.type==TYPE_TXT_VAR) {
	  if (o2.txt= (char *)malloc(PARAMLEN)) {
	    strcpy(o2.txt, var[(int)o2.num+NUMVAR].str);
	    o2.type=TYPE_TXT;
	  }
	  else {
	    print_error(error=NO_MEM_ERROR);
	    break;
	  }
	}

        strcpy(var[(int)o1.num+NUMVAR].str, o2.txt);

        p=&o2;
        ret=1;
      }
      else {
        check_delete(&o1);
        check_delete(&o2);
        print_error(error=SYNTAX_ERROR);
      }

      break;
    case (operator)or_or_eq:
    case (operator)xor_xor_eq:
    case (operator)and_and_eq:
    case (operator)or_eq:
    case (operator)xor_eq:
    case (operator)and_eq:
    case (operator)lshift_eq:
    case (operator)rshift_eq:
    case (operator)plus_eq:
    case (operator)minus_eq:
    case (operator)times_eq:
    case (operator)div_eq:
    case (operator)ampersand_eq:
      if (pop_obj(&o2) && pop_obj(&o1));
      else if (error) return 0;

      if (o2.type==TYPE_NUM_VAR) {
        o2.num = *var[(int)o2.num+NUMVAR].num;
        o2.type = TYPE_NUM;
      }

      if (o1.type==TYPE_NUM_VAR && o2.type==TYPE_NUM) {
        l=var[(int)o1.num+NUMVAR].num;

        switch (*op) {
	  case (operator)or_or_eq:  if ( o2.num) *l = 1;   break;
	  case (operator)xor_xor_eq:if ( o2.num) *l = !*l; break;
	  case (operator)and_and_eq:if (!o2.num) *l = 0;   break;
	  case (operator)or_eq:     *l  |= o2.num; break;
	  case (operator)xor_eq:    *l  ^= o2.num; break;
	  case (operator)and_eq:    *l  &= o2.num; break;
	  case (operator)lshift_eq: *l <<= o2.num; break;
	  case (operator)rshift_eq: *l >>= o2.num; break;
	  case (operator)plus_eq:   *l  += o2.num; break;
	  case (operator)minus_eq:  *l  -= o2.num; break;
	  case (operator)times_eq:  *l  *= o2.num; break;
	  case (operator)div_eq:    *l  /= o2.num; break;
	  case (operator)ampersand_eq:
	    if ((*l %= o2.num) < 0) *l += o2.num; break;
	}
        o2.num=*l;
        p=&o2;
        ret=1;
      }
      else if (*op==(operator)plus_eq && o1.type==TYPE_TXT_VAR &&
	  (o2.type==TYPE_TXT || o2.type==TYPE_TXT_VAR)) {

        if (o2.type==TYPE_TXT)
	  src=o2.txt;
        else
	  src=var[(int)o2.num+NUMVAR].str;

	i = strlen(var[(int)o1.num+NUMVAR].str);
        if (i < PARAMLEN - 1) {
	  strncpy(var[(int)o1.num+NUMVAR].str + i, src, PARAMLEN - i - 1);
	  var[(int)o1.num+NUMVAR].str[PARAMLEN - 1] = '\0';
	}
        check_delete(&o2);

        if (dst=(char *)malloc(PARAMLEN))
	  strcpy(dst, var[(int)o1.num+NUMVAR].str);
        else {
	  print_error(error=NO_MEM_ERROR);
	  break;
	}
        o1.type=TYPE_TXT;
        o1.txt=dst;

        p=&o1;
        ret=1;
      }
      else {
        check_delete(&o1);
        check_delete(&o2);
        print_error(error=SYNTAX_ERROR);
        break;
      }
      break;
    case (operator)or_or:
    case (operator)xor_xor:
    case (operator)and_and:
    case (operator)or:
    case (operator)xor:
    case (operator)and:
    case (operator)less:
    case (operator)less_eq:
    case (operator)greater:
    case (operator)greater_eq:
    case (operator)eq_eq:
    case (operator)not_eq:
    case (operator)lshift:
    case (operator)rshift:
    case (operator)minus:
    case (operator)plus:
    case (operator)times:
    case (operator)division:
    case (operator)ampersand:
      if (pop_obj(&o2) && pop_obj(&o1));
      else if (error) return 0;

      if (o1.type==TYPE_NUM_VAR) {
        o1.num = *var[(int)o1.num+NUMVAR].num;
        o1.type = TYPE_NUM;
      }
      if (o2.type==TYPE_NUM_VAR) {
        o2.num = *var[(int)o2.num+NUMVAR].num;
        o2.type = TYPE_NUM;
      }

      if (o1.type==TYPE_NUM && o2.type==TYPE_NUM) {
        if (!o2.num &&
	    (*op==(operator)division || *op==(operator)ampersand)) {
	  print_error(error=DIV_BY_ZERO_ERROR);
	  break;
	}

	switch (*op) {
	  case (operator)less:      o1.num = o1.num <  o2.num ? 1 : 0; break;
	  case (operator)less_eq:   o1.num = o1.num <= o2.num ? 1 : 0; break;
	  case (operator)greater:   o1.num = o1.num >  o2.num ? 1 : 0; break;
	  case (operator)greater_eq:o1.num = o1.num >= o2.num ? 1 : 0; break;
	  case (operator)eq_eq:     o1.num = o1.num == o2.num ? 1 : 0; break;
	  case (operator)not_eq:    o1.num = o1.num != o2.num ? 1 : 0; break;
	  case (operator)or_or:     o1.num = o1.num || o2.num; break;
	  case (operator)xor_xor:if (o2.num) o1.num = !o1.num; break;
	  case (operator)and_and:   o1.num = o1.num && o2.num; break;
	  case (operator)or:  o1.num |= o2.num; break;
	  case (operator)xor: o1.num ^= o2.num; break;
	  case (operator)and: o1.num &= o2.num; break;
	  case (operator)lshift:    o1.num <<= o2.num; break;
	  case (operator)rshift:    o1.num >>= o2.num; break;
	  case (operator)minus:   o1.num -= o2.num; break;
	  case (operator)plus:    o1.num += o2.num; break;
	  case (operator)times:   o1.num *= o2.num; break;
	  case (operator)division:o1.num /= o2.num; break;
	  case (operator)ampersand:
	    if ((o1.num %= o2.num) < 0) o1.num += o2.num; break;
	}

        p=&o1;
        ret=1;
      }
      else if ((o1.type==TYPE_TXT || o1.type==TYPE_TXT_VAR) &&
	  (o2.type==TYPE_TXT || o2.type==TYPE_TXT_VAR)) {

        if (o1.type==TYPE_TXT)
	  dst=o1.txt;
        else if (dst=(char *)malloc(PARAMLEN)) {
	  strcpy(dst, var[(int)o1.num+NUMVAR].str);
          o1.txt = dst;
	}
        else {
	  print_error(error=NO_MEM_ERROR);
	  break;
	}

        if (o2.type==TYPE_TXT)
	  src=o2.txt;
        else
	  src=var[(int)o2.num+NUMVAR].str;
 
        p=&o1;
        ret=1;
        o1.type = TYPE_NUM;
        o1.num = strcmp(dst, src);

	switch (*op) {
	  case (operator)plus: o1.type = TYPE_TXT; strcat(dst, src); break;
	  case (operator)minus:      break;
	  case (operator)less:       o1.num = o1.num <  0; break;
	  case (operator)less_eq:    o1.num = o1.num <= 0; break;
	  case (operator)greater:    o1.num = o1.num >  0; break;
	  case (operator)greater_eq: o1.num = o1.num >= 0; break;
	  case (operator)eq_eq:      o1.num = o1.num == 0; break;
	  case (operator)not_eq:     o1.num = o1.num != 0; break;
	  default:
	    print_error(error=SYNTAX_ERROR);
	    p=NULL; ret=0; break;
	}
        check_delete(&o2);
	if (*op != (operator)plus)
	  free((void *)o1.txt);
      }
      else {
        check_delete(&o1);
        check_delete(&o2);
        print_error(error=SYNTAX_ERROR);
        break;
      }
      break;
    case (operator)colon_less:
    case (operator)colon_greater:
    case (operator)less_colon:
    case (operator)greater_colon:
    case (operator)colon:
    case (operator)point_less:
    case (operator)point_greater:
    case (operator)less_point:
    case (operator)greater_point:
    case (operator)point:
      if (pop_obj(&o2) && pop_obj(&o1));
      else if (error) return 0;

      if (o2.type==TYPE_NUM_VAR) {
        o2.num = *var[(int)o2.num+NUMVAR].num;
        o2.type = TYPE_NUM;
      }

      if (o1.type!=TYPE_TXT_VAR && o1.type!=TYPE_TXT || o2.type!=TYPE_NUM) {
        check_delete(&o1);
        check_delete(&o2);
        print_error(error=SYNTAX_ERROR);
        break;
      }

      if (o2.num<=0) {
        check_delete(&o1);
        print_error(error=OUT_RANGE_ERROR);
        break;
      }

      if (o1.type==TYPE_TXT_VAR) {
        if (!(o1.txt=dst=(char *)malloc(PARAMLEN))) {
	  print_error(error=NO_MEM_ERROR);
	  break;
	}
        o1.type=TYPE_TXT;
        src=start=var[(int)o1.num+NUMVAR].str;
      }
      else {
        src=dst=start=o1.txt;
        danger=1;
      }
	   /* Potentially dangerous: start and dst are overlapping */

      switch (*op) {
	case (operator)colon_less:
	  while (o2.num) {
	    while (*src && strchr(DELIM, *src)) /*skip span of multiple DELIM*/
	      src++, j++;
	    if ((i=my_strstr(src, DELIM))!=-1)
	      o2.num--, src+=i, j+=i;
	    else break;
	  }

	  if (o2.num) { /* end of valid string before the n-th word */
	    if (!danger)
	      strcpy(dst, start);
	  }
	  else {
	    if (!danger)
	      memcpy((void *)dst, (void *)start, (size_t)j);
	    dst[j]='\0';
	  }
	  break;
	case (operator)colon:
	case (operator)colon_greater:
	  o2.num--;
	  while (*src && strchr(DELIM, *src)) /* skip span of multiple DELIM */
	    src++, j++;
	  while (o2.num) {
	    if ((i=my_strstr(src, DELIM))!=-1) {
	      o2.num--, src+=i, j+=i;
	      while (*src && strchr(DELIM, *src))
			     /* skip span of multiple DELIM */
	        src++, j++;
	    }
	    else break;
	  }

	  if (o2.num)  /* end of valid string before the n-th word */
	    *dst='\0';
	  else {
	    if (*op==(operator)colon &&
		(j=my_strstr(src, DELIM))!=-1) {
	      my_memmove((void *)dst, (void *)src, j);
	      dst[j]='\0';
	    }
	    else
	      my_memmove((void *)dst, (void *)src, 1+strlen(src));
	  }
	  break;
	case (operator)less_colon:
	  o2.num--;
	  j=strlen(src);
	  while (o2.num) {
	    while (strchr(DELIM, src[j])) /* skip span of multiple DELIM */
	      j--;
	    if ((i=my_strrstr(src, DELIM, j))!=-1)
	      o2.num--, j=i;
	    else break;
	  }

	  if (o2.num) { /* end of valid string before the n-th word */
	    if (!danger)
	      *dst='\0';
	  }
	  else {
	    my_memmove((void *)dst, (void *)start, j);
	    dst[j]='\0';
	  }
	  break;
	case (operator)greater_colon:
	  j=strlen(src)-1;
	  while (o2.num) {
	    while (strchr(DELIM, src[j])) /* skip span of multiple DELIM */
	      j--;
	    if ((i=my_strrstr(src, DELIM, j))!=-1)
	      o2.num--, j=i;
	    else break;
	  }

	  if (o2.num) { /* end of valid string before the n-th word */
	    if (!danger)
	      strcpy(dst, start);
	  }
	  else
	    my_memmove((void *)dst, (void *)(src+j+1), 1+strlen(src+j+1));
	  break;
	case (operator)point:
	  *dst++=src[o2.num-1];
	  *dst='\0';
	  break;
	case (operator)point_less:
	  j=o2.num;
	  if (!danger)
	    memcpy((void *)dst, (void *)src, (size_t)j);
	  dst[j]='\0';
	  break;
	case (operator)less_point:
	  j=strlen(src)-o2.num+1;
	  if (!danger)
	    memcpy((void *)dst, (void *)src, (size_t)j);
	  dst[j]='\0';
	  break;
	case (operator)point_greater:
	  j=o2.num-1;
	  my_memmove((void *)dst, (void *)(src+j), 1+strlen(src+j));
	  break;
	case (operator)greater_point:
	  j=strlen(src)-o2.num;
	  my_memmove((void *)dst, (void *)(src+j), 1+strlen(src+j));
	  break;
      }
      p=&o1;
      ret=1;

      break;
    case (operator)colon_question:
    case (operator)point_question:
      if (pop_obj(&o1));
      else if (error) return 0;

      if (o1.type==TYPE_TXT)
        src=o1.txt;
      else if (o1.type==TYPE_TXT_VAR)
        src=var[(int)o1.num+NUMVAR].str;
      else {
        print_error(error=SYNTAX_ERROR);
        break;
      }

      if (*op==(operator)colon_question) {
        o1.num = 0;
        while (*src && strchr(DELIM, *src)) src++;
        while (src && *src) {
	  if ((i=my_strstr(src, DELIM))!=-1) {
	    o1.num++, src+=i;
	    while (*src && strchr(DELIM, *src)) src++;
	  }
	  else {
	    src=NULL;
	    o1.num++;
	  }
	}
      }
      else
        o1.num=strlen(src);

      if (o1.type==TYPE_TXT)
        free((void *)src);

      o1.type=TYPE_NUM;
      p=&o1;
      ret=1;
      break;
    case (operator)question:
      if (pop_obj(&o2) && pop_obj(&o1));
      else if (error) return 0;

      if (o1.type==TYPE_TXT)
        dst = o1.txt;
      else if (o1.type==TYPE_TXT_VAR)
        dst = var[(int)o1.num+NUMVAR].str;
      else 
        error = SYNTAX_ERROR;

      if (!error)
        if (o2.type==TYPE_TXT)
          src = o2.txt;
        else if (o2.type==TYPE_TXT_VAR)
          src = var[(int)o2.num+NUMVAR].str;
        else 
          error = SYNTAX_ERROR;

      if (error) {
        print_error(error);
        check_delete(&o1);
        check_delete(&o2);
        break;
      }
      
      if (start = strstr(dst, src))
        i = (int)(start - dst) + 1;
      else
        i = 0;
      check_delete(&o1);
      check_delete(&o2);
      o1.type = TYPE_NUM;
      o1.num = i;
      p=&o1; ret=1;
      break;
    case (operator)null:
    case (operator)another_null:
      if (pop_obj(&o2) && pop_obj(&o1));
      else if (error) return 0;

      check_delete(&o1);
      check_delete(&o2);

      o1.type=0, o1.num=0, o1.txt=NULL;

      p=&o1;
      ret=1;
      break;
    case (operator)left_paren:
      print_error(error=MISSING_PAREN_ERROR);
      break;
    case (operator)right_paren:
      if (pop_op(op));
      else if (error) return 0;

      if (*op!=(operator)left_paren)
        print_error(error=MISMATCH_PAREN_ERROR);
      else
        ret=1;

      break;
    case (operator)_random_:
#ifdef NORANDOM
      break;
#endif
    case (operator)pre_plus:
    case (operator)pre_minus:
    case (operator)not:
    case (operator)tilde:

      if (pop_obj(&o1));
      else if (error) return 0;

      if (o1.type==TYPE_NUM_VAR) {
        o1.num = *var[(int)o1.num+NUMVAR].num;
        o1.type = TYPE_NUM;
      }
      if (o1.type==TYPE_NUM) {
	if (*op==(operator)pre_minus)
	  o1.num=-o1.num;
	else if (*op==(operator)not)
	  o1.num=!o1.num;
	else if (*op==(operator)tilde)
	  o1.num=~o1.num;
#ifndef NORANDOM
	else if (*op==(operator)_random_)
	  if (o1.num <= 0)
	    print_error(error=OUT_RANGE_ERROR);
	  else {
	    delta = LONG_MAX % o1.num;
	    while (rnd = random(), rnd > LONG_MAX - delta);
	                  /* skip numbers that would alterate distribution */
	    o1.num = rnd % o1.num;
	  }
#endif
        p=&o1;
        ret=1;
      }
      else {
        check_delete(&o1);
        print_error(error=SYNTAX_ERROR);
      }
      break;
    case (operator)star:
    case (operator)print:
      if (pop_obj(&o1));
      else if (error) return 0;
   
      if (o1.type==TYPE_NUM_VAR)
        o1.num = *var[(int)o1.num+NUMVAR].num;
      else if (o1.type==TYPE_TXT_VAR)
        o1.txt = var[(int)o1.num+NUMVAR].str;

      if (o1.type==TYPE_NUM || o1.type==TYPE_NUM_VAR) {
        if (!(o1.txt=(char *)malloc(PARAMLEN))) {
          print_error(error=NO_MEM_ERROR);
          break;
        }
	if (*op==(operator)print)
          sprintf(o1.txt, "%ld", o1.num);
        else {
          o1.txt[0] = (char)o1.num;
          o1.txt[1] = '\0';
        }
        o1.type = TYPE_TXT;
        p=&o1; ret=1;
      }
      else if (o1.type==TYPE_TXT || o1.type==TYPE_TXT_VAR) {
	if (*op==(operator)print)
          o1.num = atol(o1.txt);
        else
          o1.num = (long)(unsigned char)*o1.txt;

        check_delete(&o1);
        o1.type = TYPE_NUM;
        p=&o1; ret=1;
      }
      else 
        print_error(error=SYNTAX_ERROR);
      break;
    case (operator)pre_plus_plus:
    case (operator)post_plus_plus:
    case (operator)pre_minus_minus:
    case (operator)post_minus_minus:
      if (pop_obj(&o1));
      else if (error) return 0;

      if (o1.type==TYPE_NUM_VAR) {
        l=var[(int)o1.num+NUMVAR].num;
        o1.type=TYPE_NUM;

	if (*op==(operator)pre_plus_plus)
	  o1.num=++*l;
	else if (*op==(operator)post_plus_plus)
	  o1.num=(*l)++;
	else if (*op==(operator)pre_minus_minus)
	  o1.num=--*l;
        else
	  o1.num=(*l)--;

        p=&o1;
        ret=1;
      }
      else {
        check_delete(&o1);
        print_error(error=SYNTAX_ERROR);
      }
      break;
    case (operator)a_circle:
      if (pop_obj(&o1));
      else if (error) return 0;

      if (o1.type==TYPE_NUM_VAR) {
        o1.type=TYPE_NUM;
        o1.num=*var[(int)o1.num+NUMVAR].num;
      }

      if (o1.type==TYPE_NUM) {
        if (o1.num<-NUMVAR || o1.num>=NUMPARAM) {
	  print_error(error=OUT_RANGE_ERROR);
	  break;
	}
        o1.type=TYPE_NUM_VAR;
        p=&o1;
        ret=1;
      }
      else {
        check_delete(&o1);
        print_error(error=SYNTAX_ERROR);
      }
      break;
    case (operator)dollar:
      if (pop_obj(&o1));
      else if (error) return 0;


      if (o1.type==TYPE_NUM_VAR) {
        o1.type=TYPE_NUM;
        o1.num=*var[(int)o1.num+NUMVAR].num;
      }

      if (o1.type==TYPE_NUM) {
        if (o1.num<-NUMVAR || o1.num>=NUMPARAM) {
	  print_error(error=OUT_RANGE_ERROR);
	  break;
	}
	if (!var[(int)o1.num+NUMVAR].str) {
	  if (start = malloc(PARAMLEN)) {
	    *start = '\0';
	    var[(int)o1.num+NUMVAR].str = start;
	  }
	  else {
	    print_error(error=NO_MEM_ERROR);
	    break;
	  }
	}
        o1.type=TYPE_TXT_VAR;
        p=&o1;
        ret=1;
      }
      else {
        check_delete(&o1);
        print_error(error=SYNTAX_ERROR);
      }
      break;
    case (operator)pre_null:
    case (operator)post_null:
      ret=1;
      break;
    default:
      break;
  }

  if (!error && !ret)
    print_error(error=NOT_DONE_ERROR);
  else if (!error && p && push_obj(p));
  else if (error) return 0;

  return ret;
}

int whichfirst(op1, op2)
operator *op1, *op2;
{
  unsigned p1, p2;

  p1=op_list[*op1].priority;
  p2=op_list[*op2].priority;
  if (p1!=p2)
    return p1>p2 ? -1 : 1;

  p1 = op_list[*op1].assoc == LEFT;
  return p1 ? -1 : 1;
}

int compare_and_unload(op)
operator *op;
{
  int first=0;
  operator new;

  if (error || stk.curr_op<0)
    return 1;

  while (stk.curr_op>=0 && pop_op(&new) && !error &&
      (first = whichfirst(&new, op)) == -1 &&
      (first = 0, exe_op(&new))
    );

  if (!error)
    if (!first)
      return 1;
    else
      return push_op(&new);
  else
    return 0;
}

int eval(mindepth)
int mindepth;
{
  operator op;
  object obj;
  char endreached = 0;

  for (;;) {
    while(*line==' ') line++;
    if (!*line || *line == CMDSEP)
      endreached = 1;

    while(check_operator(LEFT, &op, mindepth) && push_op(&op) &&
	  !endreached) {
      while(*line==' ') line++;
      if (!*line || *line == CMDSEP)
	endreached = 1;
    }
    if (error) return 0;

    if (!endreached && check_object(&obj) && push_obj(&obj));
    else if (error) return 0;

    while(*line==' ') line++;
    if (!*line || *line == CMDSEP)
      endreached = 1;

    while(check_operator(RIGHT, &op, mindepth) && compare_and_unload(&op) &&
	exe_op(&op) && depth>=mindepth && !endreached) {
      while(*line==' ')
        line++;
      if (!*line || *line == CMDSEP)
        endreached = 1; 
    }
    if (error) return 0;

    if (endreached || depth < mindepth)
       break;

    if (check_operator(BINARY, &op, mindepth) &&
        compare_and_unload(&op) && push_op(&op));
    else if (error) return 0;
  }
  return 1;
}

#define PRINT_NOTHING 0
#define PRINT_AS_TEXT 1
#define PRINT_AS_NUM  2

int xeval(result, what, printmode, fromfile)
char **result, **what; int printmode, fromfile;
{
  int i=0;
  long val;
  char *txt;
  object res;

  if (!result || !*result)
    printmode = PRINT_NOTHING;

  error=0;
  is_fromfile = fromfile;
  stk.curr_obj=stk.curr_op=-1;
  line = *what;

  depth = 0;
  (void)eval(0);

  if (!error) (void)pop_obj(&res);
  if (error) tty_printf("#result not available\n");
  else if (printmode!=PRINT_NOTHING || debug) {
    if (res.type==TYPE_NUM || res.type==TYPE_NUM_VAR) {

      val = res.type==TYPE_NUM ? res.num : *var[(int)res.num+NUMVAR].num;

      if (debug) {
	PRINT_INT ("#result: %ld\n", val);
      }

      if (printmode==PRINT_AS_TEXT) {
        sprintf(*result, "%ld", val);
	i = strlen(*result);
      }
      else if (printmode==PRINT_AS_NUM) {
	*(long *)*result=val;
        i = sizeof(long);
      }
    }
    else {
      txt = res.type==TYPE_TXT ? res.txt : var[(int)res.num+NUMVAR].str;
      if (debug) {
	PRINT_INT ("#result: %s\n", txt);
      }
      if (printmode!=PRINT_NOTHING) {
	strcpy(*result, txt);
	i=strlen(txt);
      }
    }
    if (printmode!=PRINT_NOTHING)
      *result+=i;
  }
  *what=line;

  if (!error) 
    check_delete(&res);
  else {
    while (stk.curr_obj>=0) {
      pop_obj(&res);
      check_delete(&res);
    }
    res.type = 0;
  }

  if (res.type==TYPE_TXT_VAR)
    res.type = TYPE_TXT;
  else if (res.type==TYPE_NUM_VAR)
    res.type = TYPE_NUM;

  return res.type;
}
